#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This program belongs to AKKODIS INGENIERIE PRODUIT SAS.
# It is considered a trade secret, and is not to be divulged or used
# by parties who have not received written authorization from the owner.
#

_haspywin32 = False

import infinite_version
import os, sys, subprocess, glob, shutil, time, stat
import ssl
import base64
import re
import tempfile
import zipfile
import json

	
if sys.version_info[0] < 3:
	import ConfigParser
	class MyParser(ConfigParser.ConfigParser):

		def as_dict(self):
			d = dict(self._sections)
			for k in d:
				d[k] = dict(self._defaults, **d[k])
				d[k].pop('__name__', None)
			return d
else :
	import configparser
	class MyParser(configparser.ConfigParser):

		def as_dict(self):
			d = dict(self._sections)
			for k in d:
				d[k] = dict(self._defaults, **d[k])
				d[k].pop('__name__', None)
			return d
			
#https://skippylovesmalorie.wordpress.com/tag/python-windows/
def user_token_is_admin(user_token):
	import ctypes
	from ctypes import wintypes
	"""
	using the win32 api, determine if the user with token user_token has administrator rights

	See MSDN entry here: http://msdn.microsoft.com/en-us/library/aa376389(VS.85).aspx
	"""
	class SID_IDENTIFIER_AUTHORITY(ctypes.Structure):
		_fields_ = [
			("byte0", ctypes.c_byte),
			("byte1", ctypes.c_byte),
			("byte2", ctypes.c_byte),
			("byte3", ctypes.c_byte),
			("byte4", ctypes.c_byte),
			("byte5", ctypes.c_byte),
		]
	nt_authority = SID_IDENTIFIER_AUTHORITY()
	nt_authority.byte5 = 5

	SECURITY_BUILTIN_DOMAIN_RID = 0x20
	DOMAIN_ALIAS_RID_ADMINS = 0x220
	administrators_group = ctypes.c_void_p()
	if ctypes.windll.advapi32.AllocateAndInitializeSid(ctypes.byref(nt_authority), 2,
		SECURITY_BUILTIN_DOMAIN_RID, DOMAIN_ALIAS_RID_ADMINS,
		0, 0, 0, 0, 0, 0, ctypes.byref(administrators_group)) == 0:
		raise Exception("AllocateAndInitializeSid failed")

	is_admin = ctypes.wintypes.BOOL()
	if ctypes.windll.advapi32.CheckTokenMembership(
			user_token, administrators_group, ctypes.byref(is_admin)) == 0:
				raise Exception("CheckTokenMembership failed")
	ctypes.windll.advapi32.FreeSid(administrators_group)
	return is_admin.value != 0

# creates a symlink (on windows)
# http://stackoverflow.com/questions/6260149/os-symlink-support-in-windows
def symlink(source, link_name):
	os_symlink = getattr(os, "symlink", None)
	if callable(os_symlink):
		os_symlink(source, link_name)
	else:
		import ctypes
		csl = ctypes.windll.kernel32.CreateSymbolicLinkW
		csl.argtypes = (ctypes.c_wchar_p, ctypes.c_wchar_p, ctypes.c_uint32)
		csl.restype = ctypes.c_ubyte
		flags = 1 if os.path.isdir(source) else 0
		if csl(link_name, source, flags) == 0:
			raise ctypes.WinError()

#http://stackoverflow.com/questions/23598289/how-to-get-windows-short-file-name-in-python
def get_short_path_name(long_name):
	import ctypes
	from ctypes import wintypes
	_GetShortPathNameW = ctypes.windll.kernel32.GetShortPathNameW
	_GetShortPathNameW.argtypes = [wintypes.LPCWSTR, wintypes.LPWSTR, wintypes.DWORD]
	_GetShortPathNameW.restype = wintypes.DWORD
	if not os.path.exists(long_name) :
		os.makedirs(long_name)
	output_buf_size = 0
	while True:
		output_buf = ctypes.create_unicode_buffer(output_buf_size)
		needed = _GetShortPathNameW(long_name, output_buf, output_buf_size)
		if output_buf_size >= needed:
			return output_buf.value
		else:
			output_buf_size = needed
			
# tells if the user has administrative rights (required)
# http://stackoverflow.com/questions/1026431/cross-platform-way-to-check-admin-rights-in-a-python-script-under-windows
def checkAdmin () :
	import ctypes
	
	if sys.platform == 'win32':
		try:
			from win32com.shell import shellcon, shell
			raise exception('','')
			_haspywin32 = True
			is_admin = ctypes.windll.shell32.IsUserAnAdmin() != 0
		except:
			is_admin = user_token_is_admin(0) 
			#content = askInput("pywin32 is not installed, the script is not able to detect you have root access, continue ? [Y/n]")
			# content = content.lower()
			# if len(content) == 0 or (len(content) > 0 and content[0] == 'y') :
				# is_admin = True
			# else :
				#print("This script needs installation of pywin32, exiting")
				# sys.exit(1)
	else :
		is_admin = os.getuid() == 0
		
	if not is_admin :
		raise Exception("This script needs administrative rights, exiting")

def cmdToString(pCmd):
	return '"' + str('" "'.join(pCmd)) + '"'
	
# Executes a program
# returns A tuple with (stdout, stderr, error code)
def shellExecExceptOnError (cmd, disableException = False):
	stdout = ''
	stderr = ''
	errorCode = 1
	lMsg = ''
	# make sure we have a list
	if type(cmd) is str :
		import shlex
		cmd = shlex.split(cmd)
	
	try :
		proc = subprocess.Popen(cmd, 0, None, None, subprocess.PIPE, subprocess.PIPE)
		out = proc.communicate()
		if sys.version_info[0] == 3 :
			stdout = out[0].decode('utf-8', errors="ignore")
			stderr = out[1].decode('utf-8', errors="ignore")
		else :
			stdout = out[0]
			stderr = out[1]
		errorCode = proc.returncode
		if proc.returncode != 0:
			lMsg = 'Error executing %s, return %d %s %s' % (cmdToString(cmd),proc.returncode,stdout,stderr)
	except Exception as e:
		lMsg = 'Error executing %s %s' % (cmdToString(cmd),str(e))
		stdout = ''
		stderr = ''
		errorCode = 1
	except :
		lMsg = 'Error executing %s ' % (cmdToString(cmd))
		stdout = ''
		stderr = ''
		errorCode = 1
	if len(lMsg) > 0 and not disableException:
		raise Exception(lMsg)
	return (stdout, stderr, errorCode)

def setCredentials (installation,pDir) :
	
	if sys.platform == 'win32':
		cmd = [os.path.join(installation._Djuump_installers,'install form','scripts','GetAccessRights.exe'),pDir]
		# check if the folder rights are sufficient
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		if errorCode != 0:
			# change folder rights
			cmd = ['icacls',pDir,'/grant','*S-1-5-20:(OI)(CI)F','/q']
			shellExecExceptOnError(cmd)
	else :
		cmds = [
			['chown','juumpinfinite:juumpinfinite',pDir],
			['chmod','770',pDir]]
		for cmd in cmds :
			shellExecExceptOnError(cmd)

## creates the folder hierarchy
def createFolders (installation,allDirs) :
	
	print("Creating folder hierarchy")
	# common folders
	if installation._from_basepath :
		if sys.platform == 'win32':
			dst = os.path.join(installation.install_basepath,'current')
			if not os.path.exists(dst) :
				os.makedirs(dst)
		else :
			versions = os.environ['SV_VERSION_INFINITE'].split('.')
			curversion = versions[0]+'.'+versions[1]
			dst = os.path.normpath(os.path.join(installation.install_basepath,curversion))

			currentDst = os.path.join(installation.install_basepath,'current')
			if not os.path.exists(currentDst) :
				if not os.path.exists(dst) :
					os.makedirs(dst)
				olddir = os.getcwd()
				os.chdir(installation.install_basepath)
				symlink(dst,'current')
				os.chdir(olddir)
			else:
				if os.path.islink(currentDst) :
					actualDst = os.readlink(currentDst)
					if actualDst != dst :
						#this is not the correct path
						# stop services, rename folder recreatelink
						
						os.unlink(currentDst)
						
						if not os.path.exists(actualDst) :
							if not os.path.exists(dst) :
								os.makedirs(dst)
						else:
							# rename folder
							os.rename(actualDst,dst)
						olddir = os.getcwd()
						os.chdir(installation.install_basepath)
						symlink(dst,'current')
						os.chdir(olddir)
					else :
						if not os.path.exists(dst) :
							os.makedirs(dst)
		setCredentials(installation,dst)
	
	for (attrName,suffix,grant,removeDir) in allDirs:
		curDst = getattr(installation,attrName)
		dst = curDst
		if suffix != '':
			dst = os.path.join(dst,suffix)
		
		if removeDir :
			if os.path.exists(curDst) :
				print("Removing directory %s" % curDst)
				shutil.rmtree(curDst)
				time.sleep(1)
				
		if not os.path.exists(dst) :
			os.makedirs(dst)
		if dst != curDst :
			setCredentials(installation,curDst)
		setCredentials(installation,dst)
		
	if sys.platform != 'win32':
		isInArray = [item for item in allDirs if item[0] == 'ssl_folder']
		if len(isInArray)>0:
			installation._ssl_folder_pg = os.path.abspath(os.path.join(getattr(installation,'ssl_folder'),'..','ssl_pg'))
			
			if not os.path.exists(installation._ssl_folder_pg) :
				os.makedirs(installation._ssl_folder_pg)
			setCredentials(installation,installation._ssl_folder_pg)
			installation._privatekey_file_pg = os.path.join(installation._ssl_folder_pg,os.path.basename(installation.privatekey_file))
			installation._certificate_file_pg = os.path.join(installation._ssl_folder_pg,os.path.basename(installation.certificate_file))

def getFile (url):
	try:
		import urllib.request as urllib2
	except:
		import urllib2
	
	
	ctx = ssl.create_default_context()
	ctx.check_hostname = False
	ctx.verify_mode = ssl.CERT_NONE
	username = ''
	password = ''
	pattern = re.compile("(.*)/([^/:]*):([^/:]*)@(.*)")
	matchObj = pattern.match(url)
	if matchObj :
		resUrl = matchObj.group(1)+'/'+matchObj.group(4)
		username = matchObj.group(2)
		password = matchObj.group(3)
	else :
		resUrl = url
	
	request = urllib2.Request(resUrl)
	if (len(username) > 0) or (len(password) > 0) :
		creds = '%s:%s' % (username, password)
		if sys.version_info >= (3, 0):
			creds = creds.encode('ascii')
			base64string = base64.b64encode(creds)
			base64string = base64string.decode('ascii')
		else:
			base64string = base64.b64encode(creds)
		request.add_header("Authorization", "Basic %s" % base64string)
	response = urllib2.urlopen(request, context=ctx)
	
	return response.read()

def checkLocale () :
	
	if sys.platform == 'win32':
		return
	
	if not isPackageInstalled('locales') :
		installPackage('locales')
	
	cmd = ['locale','-a']
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	if errorCode != 0 :
		raise Exception('fail to retrieve locale %d %s %s'%(errorCode,stdout,stderr))
		
	lines = stdout.splitlines()
	if not 'en_US.utf8' in lines :
		src = '/etc/locale.gen'
		regContent = '^#{0,1}\\s*%s\\s+%s.*$'
		reg = (re.compile(regContent % ('en_US.UTF-8', 'UTF-8')), 'en_US.UTF-8 UTF-8')
		handle = open(src,"r")
		content = handle.read()
		handle.close()
		allLines = content.split('\n')
		newContent = ''
	
		for line in allLines :
			if reg[0].match(line) :
				newContent = newContent + reg[1] + '\n'
			else :
				newContent = newContent + line + '\n'
		handle = open(src,"w")
		handle.write(newContent)
		handle.close()
		
		cmd = ['locale-gen']
		
		shellExecExceptOnError(cmd)

def shouldUpdateFileContent(path,content):
	if not os.path.exists(path) :
		return True
	
	handle = open(path,"r")
	oldContent = handle.read()
	handle.close()
	return oldContent != content


def checkAptVersion(installation):
	import re
	lReRes = re.match(r'.*\/(.*)\/infinite\/[0-9]+.[0-9]+\/?',installation.realfusio_apt)
	if lReRes is None:
		raise Exception('Fail to analyze apt url')
	lPackagesUrl = installation.realfusio_apt
	if lPackagesUrl[-1] != '/':
		lPackagesUrl = lPackagesUrl + '/'
	lPackagesUrl = lPackagesUrl + 'dists/' + lReRes.group(1) + '/main/binary-amd64/Packages'
	
	packagelist = getFile(lPackagesUrl)
	
	lReRes = re.match(r'.*Package: lib3djuump-infinite-common.*?Version: ([0-9]+.[0-9]+.[0-9]+.[0-9]+).*',packagelist.decode('utf-8'),re.MULTILINE | re.DOTALL)
	if lReRes is None:
		raise Exception('Fail to retrieve version number from apt server')
	
	if lReRes.group(1) != os.environ['SV_VERSION_INFINITE']:
		raise Exception('This install package version (%s) is different from the package version on apt (%s)' % (os.environ['SV_VERSION_INFINITE'],lReRes.group(1)))


def configureDebianSystem(installation)	:

	install_java = hasattr(installation,'linux_oracle_java')

	main_version = re.sub(r'([0-9]+)\.([0-9]+)\..*',r'\1.\2',os.environ['SV_VERSION_INFINITE'])
	
	inFile = "/etc/apt/sources.list.d/juump-infinite-%s.list" % main_version
	content='deb %s %s main' % (installation.realfusio_apt,installation.linux_distribution)
	if shouldUpdateFileContent(inFile,content) :
		url = installation.realfusio_apt+'/3djuump-infinite.key'
		urlcontent = getFile(url)
		registerAptKey(urlcontent)
		handle = open(inFile,"w")
		handle.write(content)
		handle.close()
	
	
	inFile = os.path.join("/etc/apt/sources.list.d/pgdg.list")
	content='deb http://apt.postgresql.org/pub/repos/apt/ %s-pgdg main' % installation.linux_distribution
	if shouldUpdateFileContent(inFile,content) :
		url =  'https://www.postgresql.org/media/keys/ACCC4CF8.asc'
		urlcontent = getFile(url)
		registerAptKey(urlcontent)
		handle = open(inFile,"w")
		handle.write(content)
		handle.close()
	
	if install_java:
		inFile = os.path.join("/etc/apt/sources.list.d/elastic-6.x.list")
		content='deb https://artifacts.elastic.co/packages/6.x/apt stable main'
		if shouldUpdateFileContent(inFile,content) :
			url =  'https://artifacts.elastic.co/GPG-KEY-elasticsearch'
			urlcontent = getFile(url)
			registerAptKey(urlcontent)
			handle = open(inFile,"w")
			handle.write(content)
			handle.close()
	
		# if installation.linux_oracle_java == 'true':
			# inFile = os.path.join("/etc/apt/sources.list.d/java-ppa.list")
			# content='deb http://ppa.launchpad.net/webupd8team/java/ubuntu xenial main\ndeb-src http://ppa.launchpad.net/webupd8team/java/ubuntu xenial main'
			# if shouldUpdateFileContent(inFile,content) :
				# cmd = ['apt-key','adv','--keyserver','hkp://keyserver.ubuntu.com:80','--recv-keys','EEA14886']
				# (stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
				# if errorCode != 0 :
					# print(stdout)
					# print(stderr)
					# sys.exit(1)
				# handle = open(inFile,"w")
				# handle.write(content)
				# handle.close()
	if install_java:
		if not isPackageInstalled('apt-transport-https') :
			installPackage('apt-transport-https')
			
	print('Running apt-get update')
	
	lastfrontend = None
	if 'DEBIAN_FRONTEND' in os.environ:
		lastfrontend=os.environ['DEBIAN_FRONTEND']
	os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
	
	cmd = ['apt-get','-yq','-o','Dpkg::Options::=--force-confdef','update']
	shellExecExceptOnError(cmd)
	
	if lastfrontend is None :
		del os.environ['DEBIAN_FRONTEND']
	else :
		os.environ['DEBIAN_FRONTEND']= lastfrontend
	script =  os.path.join(installation._Djuump_installers,'install form','scripts','create_group.sh')
	os.chmod(script, stat.S_IRWXU)
	
	cmd = [script]
	shellExecExceptOnError(cmd)
	# raise Exception('YOLO')
	
	
	installPackage('ntp')
	checkLocale()

def acceptEula(installation):
	import pydoc
	
	if '--accept-eula' in sys.argv:
		return
	

	inFile = os.path.join(installation._Djuump_installers,'install form','config','eula.txt')
	handle = open(inFile,"r")
	eula = handle.read()
	handle.close()
	
	
	while (True) :
		pydoc.pager(eula)
		content = askInput('Do you accept the terms of the license ?(yes/no)')
		content = content.lower()
		if content == 'yes' :
			break;
		elif content == 'no' :
			sys.exit(1)

def configurePostgresqlCommons() :
	
	regContent = '^#{0,1}\\s*%s\\s*=.*$'
	
	src = '/etc/postgresql-common/createcluster.conf'
	reg = (re.compile(regContent % 'create_main_cluster'), "create_main_cluster = false")
			
	handle = open(src,"r")
	content = handle.read()
	handle.close()
	
	allLines = content.split('\n')
	newContent = ''
	
	for line in allLines :
		if reg[0].match(line) :
			newContent = newContent + reg[1] + '\n'
		else :
			newContent = newContent + line + '\n'

	handle = open(src,"w")
	handle.write(newContent)
	handle.close()

def registerAptKey(keyContent):
	
	(handle, tmpFile) = tempfile.mkstemp()
	os.write(handle,keyContent)
	os.close(handle)
	
	cmd = ['apt-key','add',tmpFile]
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	os.remove(tmpFile)
	if errorCode != 0 :
		print(stdout)
		print(stderr)
		raise Exception("Fail to register apt key apt-key return " + str(errorCode))

## creates the folder hierarchy
def createDirectoryFolders (installation) :
	allDirs = [ ('apache_empty_root_folder','',True,False),
				('apache_log_folder','',True,False),
				('ssl_folder','',True,False),
				('postgres_data_folder','',False,False),
				('directory_log_folder','',True,False),
				]
	createFolders(installation,allDirs)

## creates the folder hierarchy
def createProxyFolders (installation) :
	allDirs = [ ('apache_empty_root_folder','',True,False),
				('apache_log_folder','',True,False),
				('ssl_folder','',True,False),
				('postgres_data_folder','',False,False),
				('proxy_data_folder','',True,False),
				('proxy_log_folder','',True,False),
				('elastic_data_folder','',False,False),
				('elastic_log_folder','',False,False),
				('proxy_eseditor_folder','public',True,True)
				]
	createFolders(installation,allDirs)
	
## creates the ssl key certificate pair
def createSSLFiles (installation) :
	
	if hasattr(installation,'provided_certificate_file') and hasattr(installation,'provided_privatekey_file'):
		# copy provided certificate
		if not os.path.exists(installation.provided_privatekey_file):
			raise Exception('file missing : %s' % (installation.provided_privatekey_file))
		if not os.path.exists(installation.certificate_file):
			raise Exception('file missing : %s' % (installation.certificate_file))
		shutil.copy(installation.provided_privatekey_file,installation.privatekey_file)
		shutil.copy(installation.provided_certificate_file,installation.certificate_file)
		return
	
	# generate a certificate if necessary
	shouldSkip = False
	if sys.platform == 'win32' :
		shouldSkip = os.path.exists(installation.privatekey_file) and os.path.exists(installation.certificate_file)
	else :
		shouldSkip = os.path.exists(installation.privatekey_file) and os.path.exists(installation.certificate_file) and os.path.exists(installation._privatekey_file_pg) and os.path.exists(installation._certificate_file_pg)
		
	if shouldSkip :
		print("Skipping certificate creation, files already exist")
		return
	else :
		print("Creating ssl key certificate pair")
	
	
	if sys.platform == 'win32' :
		opensslFolder = os.path.join(installation._Djuump_installers,'install form','scripts')
		openssl = os.path.join(opensslFolder,'createCertificate/openssl.exe')
		opensslcnf = os.path.join(opensslFolder,'createCertificate/openssl.cnf')
		
		if not os.path.exists(opensslcnf) :
			zfile = zipfile.ZipFile(os.path.join(opensslFolder,'createCertificate.zip'))
			zfile.extractall(opensslFolder)
			if not os.path.exists(opensslcnf) :
				raise Exception('Could not find openssl conf file, exiting')
			
		os.environ["OPENSSL_CONF"] = opensslcnf.replace('/','\\')
	else :
		if not isPackageInstalled('openssl'):
			installPackage('openssl')
		openssl = 'openssl'
		
	
	
	if not os.path.exists(installation.privatekey_file) :
		cmd = [openssl,'genrsa','-out',installation.privatekey_file ,'2048']
		shellExecExceptOnError(cmd)

	if not os.path.exists(installation.certificate_file) :
		(handle,tmp_file ) = tempfile.mkstemp()
		os.close(handle)
		
		cmd = [openssl,'req','-new','-key',installation.privatekey_file,'-out',tmp_file,'-subj','/CN=%s/O=infinite/C=FR' % installation.hostname]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		if errorCode != 0 :
			os.remove(tmp_file)
			raise Exception("Error creating certificate request")

		cmd = [openssl,'x509','-req','-days','3650','-sha256','-in',tmp_file,'-signkey',installation.privatekey_file,'-out',installation.certificate_file]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		os.remove(tmp_file)
		if errorCode != 0 :
			raise Exception("Error creating certificate")
	
	if sys.platform != 'win32' :
		if os.path.exists(installation._privatekey_file_pg):
			os.remove(installation._privatekey_file_pg)
		if os.path.exists(installation._certificate_file_pg):
			os.remove(installation._certificate_file_pg)
		shutil.copy(installation.privatekey_file,installation._privatekey_file_pg)
		shutil.copy(installation.certificate_file,installation._certificate_file_pg)
		
		for item in [installation._privatekey_file_pg,
					installation._privatekey_file_pg,
					installation._ssl_folder_pg,
					installation.privatekey_file,
					installation.certificate_file,
					] :
			cmd = ['chown','-R','juumpinfinite:juumpinfinite',item]
			shellExecExceptOnError(cmd)
			
			cmd = ['chmod','-R','440',item]
			shellExecExceptOnError(cmd)
			
		for item in [installation.ssl_folder] :
			cmd = ['chown','-R','juumpinfinite:juumpinfinite',item]
			shellExecExceptOnError(cmd)
			
			cmd = ['chmod','-R','550',item]
			shellExecExceptOnError(cmd)
			
## installs postgres silently
def installPostgresImpl(installation) :
	
	if sys.platform == 'win32':
		if not installation._install_postgres :
			print("Skipping postgresql installation")
			return
		pgFolder = findInstalledProgram('Postgresql')
		if pgFolder != '':
			pgServiceName = getInstalledServiceName('postgresql-x64-')
			stopService(pgServiceName)
			# uninstall previous plugins if installed
			installedProgram = findInstalledProgram('3DJuump.*Postgres Plugins')
			if installedProgram != '' :
				uninstallProgram('3DJuump.*Postgres Plugins',['/SILENT'])
			uninstallProgram('Postgresql',['--mode','unattended'])
		
		# some times it remain some garbage in the destination folder which will prevent postgres to install
		if os.path.exists(installation.postgres_folder):
			shutil.rmtree(installation.postgres_folder,ignore_errors=True)
		
		print("Installing postgresql")
		cmd = [os.path.normpath(os.path.join(installation._Djuump_installers,'third-party',installation._postgresl_installer)),'--mode','unattended','--superaccount',installation.postgres_login,'--superpassword',installation.postgres_password, '--serverport',str(installation.postgres_port),'--prefix',os.path.normpath(installation.postgres_folder),
			'--datadir',os.path.normpath(installation.postgres_data_folder),'--locale','English, United States']
		shellExecExceptOnError(cmd)
		
	else :
		if installation._install_postgres :
			installPackage('postgresql-common')
			configurePostgresqlCommons()
		# upgrade if necessary
		packages = ['postgresql-%s' % (installation._postgresl_version)]
		installPackages(packages)
		
def installApacheImpl(installation) :
	
	if sys.platform == 'win32' :
		if not installation._install_apache :
			print("Skipping apache installation")
			return
			
	
		print("Installing apache")
		
		filepattern = os.path.join(installation._Djuump_installers,'third-party','3DJuump Infinite Apache Lounge-setup') + "*"
		allFiles = glob.glob(filepattern)
		if len(allFiles) != 1:
			print("Could not find file %s" % filepattern)
			sys.exit(1)
		
		cmd = [os.path.normpath(allFiles[0]),'/DIR=%s' % (installation.apache_folder.replace('/','\\')),'/SILENT']
		shellExecExceptOnError(cmd)
		
	else :
		# upgrade if necessary
		installPackages(['apache2'])

def getRegistryKeyValue(inputKey, keyStr) :
	value = ''
	if sys.platform == 'win32':
		if sys.hexversion < 0x03000000:
			from _winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		else :
			from winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		try :
			key = OpenKey(HKEY_LOCAL_MACHINE, inputKey,0,KEY_READ | KEY_WOW64_64KEY)
			(value,type) = QueryValueEx(key, keyStr)
		except Exception:
			pass
	return value

def duplicateRegistryKey(inputPath, outputPath, keyName) :
	if sys.hexversion < 0x03000000:
		from _winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE,KEY_WRITE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue, QueryInfoKey, CreateKey, SetValueEx, SetValue
	else :
		from winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE,KEY_WRITE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue, QueryInfoKey, CreateKey, SetValueEx, SetValue
	try:
		lKey = OpenKey(HKEY_LOCAL_MACHINE, inputPath,0,KEY_READ | KEY_WOW64_64KEY)
		(lValue,lType) = QueryValueEx(lKey, keyName)
	except:
		raise Exception('Fail to duplicate reg key %s' % inputPath)

	try:
		lNewKey = OpenKey(HKEY_LOCAL_MACHINE, outputPath,0,KEY_WRITE | KEY_WOW64_64KEY)
	except FileNotFoundError:
		lNewKey = CreateKey(HKEY_LOCAL_MACHINE,outputPath)
	except:
		raise Exception('Fail to open/create new key')
	SetValueEx(lNewKey,keyName,0,lType,lValue)

def askInput(str) :
	if '-f' in sys.argv:
		return 'y'
	if sys.hexversion < 0x03000000:
		return raw_input(str)
	else :
		return input(str)

def checkInstallationSoftwares(installation, tokens) :
	acceptEula(installation)
	
	installedStr = ''
	installed = 0
	for token,func in tokens:
		if func(installation) : 
			if installed > 0:
				installedStr = installedStr + ', ' + token
			else :
				installedStr = token
			installed = installed + 1
	
	if installed > 0 :
		if installed > 1 :
			strInput = '%s are already installed, do you want to override their configurations ? (Y/N) [Y]'
		else :
			strInput = '%s is already installed, do you want to override its configuration ? (Y/N) [Y]'
		strInput = strInput % installedStr
		goOn = askInput(strInput)
		if goOn == '' or goOn == 'y' or goOn == 'Y' :
			goOn = True
		else :
			goOn = False
			
		if not goOn:
			print('User cancelled, aborting')
			sys.exit(1)

	if sys.platform != 'win32':
		strInput = 'Do you want to run an apt_upgrade ? (Y/N) [Y]'
		goOn = askInput(strInput)
		if goOn == '' or goOn == 'y' or goOn == 'Y' :
			goOn = True
		else :
			goOn = False
		installation._run_apt_upgrade = goOn
		
def checkDirectoryInstallationSoftwares(installation, addedTokens = []) :
	if sys.platform == 'win32' :
		tokens = [('apache',checkApacheInstalled),('postgres',checkPostgresInstalled)] + addedTokens
	else :
		tokens = [('apache',checkApacheInstalled),('postgres',checkPostgresInstalled),('previous installation',checkPreviousInstallation)] + addedTokens
	
	checkInstallationSoftwares(installation,tokens)

def checkServerInstallationSoftwares(installation, addedTokens = []) :
	if sys.platform == 'win32' :
		tokens = [('apache',checkApacheInstalled),('postgres',checkPostgresInstalled),('elastic search',checkElasticInstalled)] + addedTokens
	else :
		tokens = [('apache',checkApacheInstalled),('postgres',checkPostgresInstalled),('elastic search',checkElasticInstalled),('previous folder hierarchy (will be renamed)',checkPreviousInstallation)] + addedTokens
	
	checkInstallationSoftwares(installation,tokens)

def isPackageInstalled(package) :
	from subprocess import Popen, PIPE
	isInstalled = False
	command = ['dpkg','-l',package]
	p = Popen(command, stdin=None, stdout=PIPE, stderr=PIPE)
	output, err = p.communicate()
	rc = p.returncode
	if(rc != 0)	:
		return False
	if sys.version_info[0] == 3 :
		output = output.decode('utf-8')
	content=output.splitlines()
	for inval in content :
		parsedContent = inval.split()
		if len(parsedContent) > 1:
			if (parsedContent[0] == 'ii') and (parsedContent[1] == package) :
				isInstalled = True
				break
	return isInstalled

def aptcmd(packages, interactive, word, commands) :
	
	print('%s %s' % (word, ','.join(packages)))
	
	lastfrontend = None
	if 'DEBIAN_FRONTEND' in os.environ:
		lastfrontend=os.environ['DEBIAN_FRONTEND']
	
	if not interactive :
		os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
	
	cmd = ['apt-get','-yq','-o','Dpkg::Options::=--force-confdef'] + commands + packages
	if not interactive :
		shellExecExceptOnError(cmd)
		#subprocess.call(cmd)
	else :
		errorCode = shellExecReturn(cmd)
		if errorCode != 0 :
			sys.exit(1)
	
	if not interactive :
		if lastfrontend is None :
			del os.environ['DEBIAN_FRONTEND']
		else :
			os.environ['DEBIAN_FRONTEND']= lastfrontend

def installPackages(packages, interactive=False,hold=False) :
	aptcmd(packages,interactive,'Installing', ['install'])
	if hold :
		for item in packages :
			cmd = ['apt-mark','hold', item]
			shellExecExceptOnError(cmd)

def uninstallPackages(packages, interactive=False) :
	aptcmd(packages,interactive,'Uninstalling', ['remove','--purge'])
			
def installPackage(package, interactive=False, hold=False) :
	installPackages([package],interactive,hold)

def uninstallPackage(package, interactive=False) :
	uninstallPackages([package],interactive)

def searchForExistingApacheCusto(installation):
	print("Checking for apache conf custo (Include directives)")

	lInfiniteConfFile = None
	lHttpdConfFile = None
	if sys.platform == 'win32':
		# for windows look for includes in both infinite conf file and base httpd.conf
		# for windows we also need to look for an older version 
		if not os.path.exists(installation.apache_folder):
			return
		
		lApacheFolders = ['Apache'+os.environ['SV_VERSION_APACHE']] + sorted(os.listdir(installation.apache_folder),reverse=True)
		for f in lApacheFolders:
			lTmp1 = os.path.join(installation.apache_folder,f,'conf','httpd.conf')
			lTmp2 = os.path.join(installation.apache_folder,f,'conf',installation.apache_conf_file_name+'.conf')	
			if os.path.exists(lTmp1) and os.path.exists(lTmp2):
				lInfiniteConfFile = lTmp2
				lHttpdConfFile = lTmp1
				break
	else:
		# for linux look only for includes in infinite conf file
		lInfiniteConfFile = os.path.join('/etc/apache2/sites-available',installation.apache_conf_file_name+'.conf')

	lIncludeRe = re.compile(r'^\s*Include\s+(.*)$',re.MULTILINE)
	if not lHttpdConfFile is None:
		lOriginalConfFile = ''
		with open(lHttpdConfFile,encoding='utf-8') as f:
			lOriginalConfFile = f.read()
		for i in lIncludeRe.findall(lOriginalConfFile):
			if 'conf/extra/proxy-html.conf' in i:
				continue
			elif 'conf/3djuump' in i and ('proxy' in i or 'directory' in i or 'server' in i):
				continue
			installation._httpd_conf_custom_includes.append(i)
	
	lHttpsIncludes = []
	if not lInfiniteConfFile is None and os.path.isfile(lInfiniteConfFile):
		lOriginalConfFile = ''
		with open(lInfiniteConfFile,encoding='utf-8') as f:
			lOriginalConfFile = f.read()
		# extract https vhost
		lHttpsVHostRe = re.compile('<VirtualHost _default_:%s>.*?<\/VirtualHost>' % (installation.apache_https_port),re.DOTALL)
		lMatch = lHttpsVHostRe.search(lOriginalConfFile)
		if not lMatch is None:
			for i in lIncludeRe.findall(lMatch.group(0)):
				lHttpsIncludes.append('Include ' + i)
	
	if len(installation._httpd_conf_custom_includes) > 0:
		print('Found Include to restore in httpd.conf ' + str(installation._httpd_conf_custom_includes))
	if len(lHttpsIncludes) > 0:
		installation._httpd_infinite_conf_custom_includes = '\n\t'.join(lHttpsIncludes)
		print('Found Include to restore in %s.conf ' % installation.apache_conf_file_name + str(lHttpsIncludes))

def checkApacheInstalled(installation) : 
	searchForExistingApacheCusto(installation)
	
	print("Checking apache installation")
	
	if sys.platform == 'win32':
		installerPath = getRegistryKeyValue(r'SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\{E42D260D-FDE9-459A-A20A-19B64A0917DC}_is1', 'InstallLocation')
		installation._install_apache = True
		if installerPath != '' :
			binExe = os.path.join(installerPath,'Apache'+os.environ['SV_VERSION_APACHE'],'bin','httpd.exe')
			installation._install_apache = not os.path.exists(binExe)
		return not installation._install_apache
	else:
		installation._install_apache = (not isPackageInstalled('apache2')) or (not isPackageInstalled('libapache2-mod-fcgid'))
		return not installation._install_apache

def checkPreviousInstallation(installation) :
	print('checking previous installation')
	if sys.platform == 'win32':
		return False
	
	# common folders
	if installation._from_basepath :
		versions = os.environ['SV_VERSION_INFINITE'].split('.')
		curversion = versions[0]+'.'+versions[1]
		dst = os.path.normpath(os.path.join(installation.install_basepath,curversion))
		link = os.path.join(installation.install_basepath,'current')
		if os.path.exists(link) :
			if os.path.islink(link) :
				actualDst = os.readlink(link)
				if actualDst == dst :
					#this is already the correct path
					return False
			return True
	return False


def getImagePathServiceLocation(serviceName) :
	value = ''
	if sys.platform == 'win32':
		if sys.hexversion < 0x03000000:
			from _winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		else :
			from winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		try :
			servicestr = r'SYSTEM\CurrentControlSet\Services' + '\\' + serviceName
			curkey = OpenKey(HKEY_LOCAL_MACHINE, servicestr,0,KEY_READ | KEY_WOW64_64KEY)
			(command,type) = QueryValueEx(curkey, 'ImagePath')
			if len(command) > 0:
				value = command
		except Exception:
			pass
	return value

def getWindowsPostgresqlDataClusterLocation(serviceName) :
	if sys.platform != 'win32':
		return ''
	lFullLocation = getImagePathServiceLocation(serviceName)
	regObj = re.compile(r'-D\s+(?:(?:"([^"]+)")|(\S+))')
	match = regObj.search(lFullLocation)
	if not match is None:
		value = match.group(1)
	if len(value) == 0:
		raise Exception('Cannot find the location of the installed postgresql cluster')
	return value
	
def getInstalledServiceLocation(serviceName) :
	value = ''
	if sys.platform == 'win32':
		command = getImagePathServiceLocation(serviceName)
		if len(command) == 0:
			return value
		if command[0] == '\"':
			regObj = re.compile(r'^"([^"]+)"')
		else :
			regObj = re.compile(r'^([^\s]+)(\s|$)')
		match = regObj.search(command)
		if not match is None:
			value = match.group(1)
			value = os.path.split(value)[0]
	return value

def getPreviousPostgresWinMainVersion() :
	if sys.platform != 'win32':
		return ''
	serviceName = getInstalledServiceName('postgresql-x64-')
	if serviceName == '' :
		raise Exception("Internal error, cannot find main postgres version")
	reg = re.compile("postgresql-x64-(.*)\\s*")
	regObj = reg.match(serviceName)
	if regObj is None :
		raise Exception("Internal error, cannot find main postgres version")
	return regObj.group(1)

def getInstalledServiceName(pattern) :
	value = ''
	if sys.platform == 'win32':
		
		if sys.hexversion < 0x03000000:
			from _winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		else :
			from winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		try :
			regObj = re.compile(pattern,re.IGNORECASE)
			servicestr = r'SYSTEM\CurrentControlSet\Services'
			curkey = OpenKey(HKEY_LOCAL_MACHINE, servicestr,0,KEY_READ | KEY_WOW64_64KEY)
			count = 0
			try :
				while 1:
					name = EnumKey(curkey, count)
					if not regObj.search(name) is None :
						if value != '' :
							print(value)
							print("Multiple installations found !")
							return ''
						value = name
					count = count + 1
			except WindowsError:
				pass
		except Exception:
			pass
	return value	

def getVersionVal (pString) :
	fullVersion = 0
	pString = pString.replace('_','.')
	all_vers = pString.split('.')
	multiplicator = 1000
	for i in range(0,len(all_vers)) :
		fullVersion += int(all_vers[i]) * multiplicator
		multiplicator = multiplicator / 1000
	return fullVersion

def getPreviousPostgresVersion ():
	if sys.platform == 'win32' :
		installLocation = findInstalledProgram('Postgresql')
		if installLocation == '' :
			return ''
		binExe = os.path.join(installLocation,'bin','pg_ctl.exe')
		if not os.path.exists(binExe) :
			return ''
		cmd = [binExe,'--version']
	else :
		cmd = ['pg_config','--version']
	
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	if errorCode == 0:
		versionfull = stdout
		regObj = re.compile(r'[^0-9]*([0-9\.]+)(?:[^0-9\.].*)?$')
		regVal = regObj.match(versionfull)
		if not regVal is None:
			return regVal.group(1)
			# we assume 
	#print('cannot get pg version')
	return ''
	
def getInstalledProgramProperty(pattern,key) :
	value = ''
	if sys.platform == 'win32':
		
		if sys.hexversion < 0x03000000:
			from _winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		else :
			from winreg import OpenKey, QueryValueEx, CloseKey, HKEY_LOCAL_MACHINE, KEY_READ, KEY_WOW64_64KEY, EnumKey, QueryValue
		try :
			regObj = re.compile(pattern,re.IGNORECASE)
			uninstallstr = r'SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall'
			curkey = OpenKey(HKEY_LOCAL_MACHINE, uninstallstr,0,KEY_READ | KEY_WOW64_64KEY)
			count = 0
			try :
				while 1:
					name = EnumKey(curkey, count)
					program = uninstallstr + '\\' + name
					try :
						programkey = OpenKey(HKEY_LOCAL_MACHINE, program,0,KEY_READ | KEY_WOW64_64KEY)
						(displayname,type) = QueryValueEx(programkey, 'DisplayName')
						if not regObj.search(displayname) is None :
							if value != '' :
								print(value)
								print(displayname)
								print("Multiple postgres installations found !")
								return ''
							(value,type) = QueryValueEx(programkey, key)
					except Exception:
						pass
					count = count + 1
			except WindowsError:
				pass
		except Exception:
			pass
	return value
	
def findInstalledProgram(pattern) :
	installLocation = getInstalledProgramProperty(pattern,'InstallLocation')
	return installLocation.strip('"')
	
def uninstallProgram(pattern, args = []) :
	import shlex
	uninstallStr = getInstalledProgramProperty(pattern,'UninstallString')
	if uninstallStr == '':
		raise Exception("cannot find uninstall procedure for program %s"% pattern)
	# shellex is used to split args, and '\' is not handled properly => replace \ by / (fortunatly, \ will
	# only be present on the first arg
	uninstallStr = uninstallStr.replace('\\','/')
	cmd = shlex.split(uninstallStr) + args
	print("Uninstalling %s" % pattern)
	shellExecExceptOnError(cmd)

def getElasticSearchVersions () :
	if sys.platform != 'win32' :
		return (0,0)
		
	# find the command line for the elastic service, we will get the bin folder
	installLocation = getInstalledServiceLocation('elasticsearch-service-x64')
	javaversion = 0
	elasticversion = 0
	if installLocation == '' :
		# bail out, not found
		return (javaversion,elasticversion)
	
	# get elastic installation folder with JAVA_HOME
	serviceFile = os.path.join(installLocation,'service.bat')
	handle = open(serviceFile,"r")
	content = handle.read()
	handle.close()
	reg = re.compile("set\s+JAVA_HOME\s*=(.*)\n")
	match = reg.search(content)
	if not match is None :
		javaFolder = match.group(1).strip()
		# set the java_home in env
		os.environ['JAVA_HOME'] = javaFolder
	
	# execute elasticsearch --version
	cmd = [os.path.join(installLocation,'elasticsearch.bat'),'--version']
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	if errorCode != 0 :
		# something is broken in the realm
		return (javaversion,elasticversion)
	stdout = stdout.strip()
	
	# parse output to get java and elastic versions
	reg = re.compile("Version\s*:\s*([0-9\.]+)[^0-9\.]")
	match = reg.search(stdout)
	if not match is None :
		elasticversion = getVersionVal(match.group(1))
	reg = re.compile("JVM\s*:\s*([0-9\._]+)\s*$")
	match = reg.search(stdout)
	if not match is None :
		javaversion = getVersionVal(match.group(1))
	return (javaversion,elasticversion)	
	
def getElasticSearchFolders () :
	if sys.platform != 'win32' :
		return ('','')
		
	# find the command line for the elastic service, we will get the bin folder
	installLocation = getInstalledServiceLocation('elasticsearch-service-x64')
	javalocation = ''
	elasticlocation = ''
	if installLocation == '' :
		# bail out, not found
		return (javalocation,elasticlocation)
	
	elasticlocation = os.path.split(installLocation)[0]
	# get elastic installation folder with JAVA_HOME
	serviceFile = os.path.join(installLocation,'service.bat')
	handle = open(serviceFile,"r")
	content = handle.read()
	handle.close()
	reg = re.compile("set\s+JAVA_HOME\s*=(.*)\n")
	match = reg.search(content)
	if not match is None :
		javalocation = match.group(1).strip()
	return (javalocation,elasticlocation)
	
def checkElasticInstalled(installation) :
	print("Checking elastic search installation")
	if sys.platform == 'win32':
		expectedJava = '1.8.0_%s' % os.environ['SV_VERSION_JRE']
		expectedJavaVersion = getVersionVal(expectedJava)
		expectedElasticVersion = getVersionVal(os.environ['SV_VERSION_ES_FULL'])
		(javaversion,elasticversion) = getElasticSearchVersions()
		installation._install_elastic = ((expectedElasticVersion > elasticversion) or (expectedJavaVersion > javaversion))
	else :
		installation._install_elastic = (not isPackageInstalled('elasticsearch'))
		if installation.linux_oracle_java == True:
			cmd=['java','-version']
			(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
			if errorCode != 0 :
				print("Java is not installed on your system, please download the oracle java version and retry the installation")
				sys.exit(1)
			
			reg = re.compile(r"Java\(TM\)")
			match = reg.search(stdout+"\n"+stderr)
			if match is None :
				print("Java is installed on your system, but not the Oracle one, please download the oracle java version and retry the installation")
				sys.exit(1)
	return not installation._install_elastic

def checkPostgresInstalled(installation) :
	
	print("Checking postgres installation")
	
	postgresversion = getPreviousPostgresVersion()
	
	if postgresversion == '' :
		# postgres is not installed
		#print("postgresql is not installed")
		installation._install_postgres = True
	else :
		if sys.platform == 'win32' :
			installation._previous_win_postgres_folder = findInstalledProgram('Postgresql')
			installation._previous_postgres_version = getPreviousPostgresWinMainVersion()
			installation._previous_win_postgres_data_folder = getWindowsPostgresqlDataClusterLocation(getPreviousPostgresqlServiceName(installation))
			
			expectedpostgresversion = getVersionVal(os.environ['SV_VERSION_PG_FULL'])
			installation._install_postgres = (expectedpostgresversion > getVersionVal(postgresversion))
			if not installation._install_postgres:
				installation.postgres_folder =	installation._previous_win_postgres_folder
				installation.postgres_data_folder = installation._previous_win_postgres_data_folder
		else:
			old_versions = getInstalledClusterVersions(installation._postgresl_cluster_name)
			if len(old_versions) > 1 :
				raise Exception("Multiple postgresql versions found (more than one), this script cannot handle this case")
			if len(old_versions) == 1 :
				installation._previous_postgres_version = old_versions[0]
			installation._install_postgres = (not isPackageInstalled('postgresql-%s'%(installation._postgresl_version)))
	
		checkHasBuilds(installation)
	
	return not installation._install_postgres
	
def installPostgresPluginsImpl(installation):
	print("Installing postgres plugins")
	
	if sys.platform != 'win32':
		# plugins are installed automatically due to the dependency system
		return
		
	stopService(getPostgresqlServiceName(installation));
	
	exe = os.path.join(installation._Djuump_installers,'dist','win',installation._plugin_installer)
	allFiles = glob.glob(exe+"*")
	if len(allFiles) != 1 :
		print("Error finding postgres plugins")
		sys.exit(1)
	exe = allFiles[0]
	
	cmd = [os.path.normpath(exe),'/DIR=%s' % (installation.infinite_postgres_plugins_folder.replace('/','\\')),'/SILENT']
	shellExecExceptOnError(cmd)
	
	
	restartService(getPostgresqlServiceName(installation),altName='postgresql');

def installApachePluginsWin (installation) :
	if sys.platform != 'win32' :
		return
	
	print("Installing apache plugins")
	
	stopService("Apachehttp3djuumpInfinite")
	exe = os.path.join(installation._Djuump_installers,'dist','win',installation._apache_plugin_installer)
	allFiles = glob.glob(exe+"*")
	if len(allFiles) != 1 :
		print("Error finding apache plugins")
		sys.exit(1)
	exe = allFiles[0]
	
	(handle,tmp_file) = tempfile.mkstemp(suffix='.ini')
	os.close(handle)
	saveIni({'General':{'ApacheInstallFolder': os.path.join(installation.apache_folder,'Apache'+os.environ['SV_VERSION_APACHE'])}},tmp_file)
	
	
	cmd = [os.path.normpath(exe),'/SILENT','/LOADINF=%s' % tmp_file]
	shellExecExceptOnError(cmd)
	
def configureString(installation,content) :
	inAttr = getAttributes(installation)
	lRe = re.compile(r'.*\${([A-Za-z0-9-_]+)}.*',re.MULTILINE | re.DOTALL)
	while True:
		lMatchRes = lRe.match(content)
		if lMatchRes is None:
			break
		if not hasattr(installation,lMatchRes.group(1)):
			raise Exception('Missing value %s' % (lMatchRes.group(1)))
		content = content.replace('${'+lMatchRes.group(1)+'}',str(getattr(installation,lMatchRes.group(1))))
	return content

def copyAndConfigure(installation,src,dst, replaceString, compareContent = None) :
	
	handle = open(src,"r")
	content = handle.read()
	handle.close()
	
	content = configureString(installation,content)
	for (find,rep) in replaceString:
		content = content.replace(find,rep)
	
	if os.path.exists(dst) :
		handle =  open(dst,"r")
		oldContent = handle.read()
		handle.close()
		if compareContent != None :
			if compareContent(oldContent,content):
				return False
			
		if oldContent == content :
			return False
			
	handle = open(dst,"w")
	handle.write(content)
	handle.close()
	return True

def isProcessRunning(name) :
	cmd =['ps','cax']
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd)
	
	alllines=stdout.splitlines()
	for line in alllines:
		if name in line:
			return True
	return False

def configureApacheImpl (installation, modules,ports, filename, shouldLog=True) :
	import random
	
	lApacheReplaceStrings = []
	
	lModulesToDisable = [('access_compat','modules/mod_access_compat.so')]
	lApachePwdExe = None
	if sys.platform == 'win32':
		if shouldLog:
			print("Configuring apache")
			
		src = os.path.join(installation.apache_folder,'Apache'+os.environ['SV_VERSION_APACHE'],'conf','httpd.conf')
		dst = src
		
		lApachePwdExe = os.path.join(installation.apache_folder,'Apache'+os.environ['SV_VERSION_APACHE'],'bin','htpasswd.exe')
		
		srcTemplate = os.path.join(installation._Djuump_installers,'install form','config','httpd_win.conf')
	
		bakupFile = os.path.join(os.getcwd(),os.path.split(dst)[1]+".bak")
		if not os.path.exists(bakupFile):
			shutil.copy(dst,bakupFile)
	
		handle = open(srcTemplate,"r")
		content = handle.read()
		handle.close()
	
	
		content = configureString(installation,content)
		enableregs = []
		for (moduleName,moduleSo) in modules :
			enableregs.append(re.compile('^#(LoadModule\\s+%s_module\\s+%s.*)' % (moduleName,moduleSo)))
		
		disableregs = []
		for (moduleName,moduleSo) in lModulesToDisable :
			disableregs.append(re.compile('^\\s*(LoadModule\\s+%s_module\\s+%s.*)' % (moduleName,moduleSo)))
	
		allLines = content.split('\n')
		newContent = ''
		for line in allLines :
			if line != '' :
				foundContent = False
				for reg in enableregs :
					matchObj = reg.match(line)
					if matchObj :
						foundContent = True
						newContent = newContent + matchObj.group(1) + '\n'
						break
				for reg in disableregs :
					matchObj = reg.match(line)
					if matchObj :
						foundContent = True
						newContent = newContent + '#' + matchObj.group(1) + '\n'
						break
				if not foundContent:
					newContent = newContent + line + '\n'
				
		newContent = newContent + ('Include "conf/%s.conf"\n' % filename)
		for i in installation._httpd_conf_custom_includes:
			newContent = newContent + ('Include %s\n' % i)
		handle = open(dst,"w")
		handle.write(newContent)
		handle.close()
		
		src = os.path.join(installation._Djuump_installers,'install form','config',filename+'_windows.conf')
		dst = os.path.join(installation.apache_folder,'Apache'+os.environ['SV_VERSION_APACHE'],'conf',filename+'.conf')
		copyAndConfigure(installation,src,dst,lApacheReplaceStrings)
		
		restartService("Apachehttp3djuumpInfinite",altName='apache2')
	else :
		restartApache = False
		if shouldLog:
			print("Configuring Apache")
		
		lApachePwdExe = 'htpasswd'
		
		if len(modules) > 0:
			cmd = ['a2enmod']
			for module in modules :
			# mod_version is statically linked in Linux, not external module
			# do not try to enable it
				if module[0] != 'version' :
					cmd.append(module[0])
			shellExecExceptOnError(cmd)
		
		# disable some modules
		cmd = ['a2dismod']+[module[0] for module in lModulesToDisable]
		shellExecExceptOnError(cmd)
		
		src = '/etc/apache2/envvars'
		
		handle = open(src,"r")
		content = handle.read()
		handle.close()
		
		regs = []	
		regs.append((re.compile('^\\s*export\\s+APACHE_RUN_USER.*'),'export APACHE_RUN_USER=juumpinfinite'))
		regs.append((re.compile('^\\s*export\\s+APACHE_RUN_GROUP.*'),'export APACHE_RUN_GROUP=juumpinfinite'))
		allLines = content.splitlines()
		
		newContent = ''
		count = 0
		for line in allLines :
			if line != '' :
				foundContent = False
				for (reg,replacement) in regs :
					matchObj = reg.match(line)
					if matchObj :
						count = count + 1
						foundContent = True
						newContent = newContent + replacement + '\n'
						break
				if not foundContent:
					newContent = newContent + line + '\n'
		
		if count != len(regs) :
			print("Error configuring apache envvars")
			sys.exit(1)
		
		if newContent != content :
			restartApache = True
			handle = open(src,"w")
			handle.write(newContent)
			handle.close()
		
		content = ''
		for item in ports :
			content = content + 'Listen %s\n' % item
		
		src = '/etc/apache2/ports.conf'
		
		oldContent = ''
		if os.path.exists(src) :
			handle = open(src,"r")
			oldContent = handle.read()
			handle.close()
		
		if oldContent != content :
			restartApache = True
			handle = open(src,"w")
			handle.write(content)
			handle.close()
		
		src = os.path.join(installation._Djuump_installers,'install form','config',filename+'_linux.conf')
		dst = os.path.join('/etc/apache2/sites-available',filename+'.conf')
		if copyAndConfigure(installation,src,dst,lApacheReplaceStrings) :
			restartApache = True
			
		allSites = glob.glob("/etc/apache2/sites-enabled/*")
		for item in allSites :
			# only disable infinite related sites
			if not item.startswith('3djuump'):
				print('keep apache site enabled : %s' % (item))
				continue
			cmd = ['a2dissite',os.path.split(item)[1] ]
			shellExecExceptOnError(cmd)
		
		cmd = ['a2ensite',filename+'.conf']
		shellExecExceptOnError(cmd)
		
		if not restartApache :
			restartApache = not isProcessRunning('apache2')
			
		if restartApache :
			restartService('apache2')
	
	# store admin password
	shellExecExceptOnError([lApachePwdExe,'-bcB',installation.auth_pwd_file,installation.postgres_login,installation.postgres_password])
	if sys.platform != 'win32':
		shellExecExceptOnError(['chown','juumpinfinite:juumpinfinite',installation.auth_pwd_file])
		shellExecExceptOnError(['chmod','440',installation.auth_pwd_file])
	
def configureApache (installation) :
	
	modules = [
				('proxy','modules/mod_proxy.so'),
				('proxy_http','modules/mod_proxy_http.so'),
				('socache_shmcb','modules/mod_socache_shmcb.so'),
				('mime','modules/mod_mime.so'),
				('setenvif','modules/mod_setenvif.so'),
				('ssl','modules/mod_ssl.so'),
				('rewrite','modules/mod_rewrite.so'),
				('headers','modules/mod_headers.so'),
				('infinite','modules/mod_infinite.so'),
				('authz_core','modules/mod_authz_core.so'),
				('authz_host','modules/mod_authz_host.so'),
				('version','modules/mod_version.so'),
				]
				
		
	ports = [80,'%s'%installation.apache_https_port]
	configureApacheImpl(installation,modules,ports,installation.apache_conf_file_name)

def configureInfiniteService(installation, pServiceName,pCreateDbCommand):
	import urllib
	if sys.platform == 'win32':
		lConfFile = os.path.join(getAppData(),pServiceName,'conf.json')
	else:
		lConfFile = os.path.join('/etc',pServiceName,'conf.json')
		
	#load existing configuration
	lConf = loadJson(lConfFile)
	
	lIsDirectory = 'directory' in pServiceName
	
	
	lPostgres = lConf['postgres']
	lPostgres['login'] = installation.postgres_login
	lPostgres['password'] = installation.postgres_password
	
	lConf['verify_ssl_peer'] = installation.verify_ssl_peer
	
	if sys.platform != 'win32':
		lConf['serviceuser'] = 'juumpinfinite'
	
	lLokiHostLabel = ''
	
	lServiceExe = None
	if lIsDirectory:
		
		if sys.platform == 'win32':
			lServiceExe = os.path.join(installation.directory_binary_folder,installation.directory_exe_name)
		else:
			lServiceExe = os.path.join('/usr/lib/',installation.directory_service_name,'bin',installation.directory_exe_name)
			
		lConf['log']['file'] = os.path.join(installation.directory_log_folder,pServiceName+".txt").replace('\\','/')
		
		lPostgres['writeurl'] = 'postgres://127.0.0.1:%s/InfiniteDirectory' % (installation.postgres_port)
		lPostgres['readurl'] = None
		
		lDirectoryApi = lConf['directoryapi']
		lDirectoryApi['bindport'] = installation.directory_api_local_port
		lDirectoryLocalUrl = urllib.parse.urlparse(installation.directory_public_url)
		lDirectoryLocalUrl = lDirectoryLocalUrl._replace(netloc='127.0.0.1')
		lDirectoryApi['localurl'] = urllib.parse.urlunparse(lDirectoryLocalUrl)
		lDirectoryApi['publicurl'] = installation.directory_public_url

		if hasattr(installation,'lmx_server') and hasattr(installation,'lmx_feature'):
			if not 'lmx' in lConf or lConf['lmx'] is None:
				lConf['lmx'] = {}
			lLmx = lConf['lmx']
			lLmx['server'] = installation.lmx_server
			lLmx['license'] = installation.lmx_feature
		else:
			lConf['lmx'] = None
		
		
		lConf['openidconnect'] = installation.openidconnect
		# strip "offline install" attribute
		if 'jwks_uri' in lConf['openidconnect']:
			del lConf['openidconnect']['jwks_uri']
	else:
	
		if sys.platform == 'win32':
			lServiceExe = os.path.join(installation.proxy_binary_folder,installation.proxy_exe_name)
		else:
			lServiceExe = os.path.join('/usr/lib/',installation.proxy_service_name,'bin',installation.proxy_exe_name)
			
		lConf['log']['file'] = os.path.join(installation.proxy_log_folder,pServiceName+".txt").replace('\\','/')
		
		lPostgres['url'] = 'postgres://127.0.0.1:%s/InfiniteProxy' % (installation.postgres_port)
		
		lDirectoryApi = lConf['directoryapi']
		lDirectoryApi['login'] = installation.directory_postgres_login
		lDirectoryApi['password'] = installation.directory_postgres_password
		lDirectoryApi['url'] = installation.directory_public_url
		
		lConf['elasticport'] = installation.elastic_port
		
		lConf['generatewebdataonnextrestore'] = installation.generate_web_data
		
		lProxyApi = lConf['proxyapi']
		lProxyApi['bindport'] = installation.proxy_api_local_port
		lProxyLocalUrl = urllib.parse.urlparse(installation.proxy_public_url)
		lProxyLocalUrl = lProxyLocalUrl._replace(netloc='127.0.0.1')
		lProxyApi['localurl'] = urllib.parse.urlunparse(lProxyLocalUrl)
		lProxyApi['publicurl'] = installation.proxy_public_url

		lConf['workingfolder'] = installation.proxy_data_folder
		
	if hasattr(installation,'loki_post_url') and not installation.loki_post_url is None:
		if not 'loki' in lConf['log'] or lConf['log']['loki'] is None:
			lConf['log']['loki'] = {"label":{}}
		lLokiPostUrl = urllib.parse.urlparse(installation.loki_post_url)
		if not lLokiPostUrl.username is None:
			lConf['log']['loki']['login'] = lLokiPostUrl.username
			lConf['log']['loki']['password'] = lLokiPostUrl.password
		else:
			for k in ['login','password']:
				if k in lConf['log']['loki']:
					del lConf['log']['loki'][k]
		# remove credentials from url
		lLokiPostUrl = lLokiPostUrl._replace(netloc=lLokiPostUrl.hostname)
		lConf['log']['loki']['posturl'] = urllib.parse.urlunparse(lLokiPostUrl)
		lConf['log']['loki']['label']['host'] = installation.hostname
		lConf['log']['loki']['verify_ssl_peer'] = installation.verify_ssl_peer
	elif 'loki' in lConf['log']:
		del lConf['log']['loki']

		
		
	saveJson(lConf,lConfFile)
	
	
	# consolidate configuration
	shellExecExceptOnError([lServiceExe,'-consolidateconfiguration'])
	
	# create db
	shellExecExceptOnError([lServiceExe] + pCreateDbCommand)
	
	# if sys.platform == 'win32':
		# enable the service
		
def registerDefaultApplications(installation):
	import requests, urllib
	lHttpPool = requests.Session()
	if not installation.verify_ssl_peer:
		lHttpPool.verify = False
		requests.packages.urllib3.disable_warnings(requests.packages.urllib3.exceptions.InsecureRequestWarning)
	lApiKey = base64.b64encode((installation.postgres_login + ':' + installation.postgres_password).encode('utf-8')).decode('ascii')
	lHeaders = {'x-infinite-apikey':lApiKey, 'Authorization': 'Basic ' + lApiKey}
	lResponse = lHttpPool.post(installation.directory_public_url + '/api/manage/applications',headers = lHeaders,proxies={'https':None})
	
	if lResponse.status_code != 200:
		raise Exception('%s/api/manage/applications returns %d, %s, %s' % (installation.directory_public_url,lResponse.status_code,lResponse.text,lApiKey))
	lExistingApps = {}
	lRegisteredUrls = set()
	for a in lResponse.json()['data']:
		lExistingApps[a['name']] = a['redirecturls']
		lRegisteredUrls.update(a['redirecturls'])
	
	lAppToRegister = [
			('3D Juump Infinite Admin Front End','3D Juump Infinite administration front-end page', installation.directory_public_url + '/frontend/'),
			('3D Juump Infinite Native Client','3D Juump Infinite native client application', installation.directory_public_url + '/api/static/onauthenticationdone.html'),
			('3D Juump Infinite Home','3D Juump Infinite home page', installation.directory_public_url + '/home/'),
			('3D Juump Infinite Web Client','3D Juump Infinite web client application', installation.directory_public_url + '/webclient/'),
			]

	for (name,desc, url) in lAppToRegister:
		lUrls = []
		if not url in lRegisteredUrls:
			lUrls.append(url)
		if ':443' in url:
			lAltUrl = url.replace(':443','')
			if not lAltUrl in lRegisteredUrls:
				lUrls.append(lAltUrl)
		if len(lUrls) == 0:
			continue
		lPutBody = {}
		if name in lExistingApps:
			# update existing app
			lUrls = lUrls + lExistingApps[name]
		else:
			# create the app
			lPutBody['description'] = desc
		lPutBody['redirecturls'] = lUrls
		lHttpPool.put(installation.directory_public_url + '/api/manage/applications/' + name,json=lPutBody,headers = lHeaders,proxies={'https':None})

def isClusterExists(installation):
	if sys.platform == 'win32' :
		return True
	cmd = ['pg_lsclusters']
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd)
	
	targetVersion = installation._postgresl_version
	
	lines = stdout.splitlines()
	for line in lines :
		allContent = line.split()
		if len(allContent) > 2 :
			if (allContent [0] == targetVersion) and (allContent [1] == installation._postgresl_cluster_name) :
				return True
	return False

def getInstalledClusterVersions (installation_cluster_name) :
	if sys.platform == 'win32' :
		return []
	cmd = ['pg_lsclusters']
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd)
	
	versions = []
	lines = stdout.splitlines()
	for line in lines :
		allContent = line.split()
		if len(allContent) > 2 :
			if allContent [1] == installation_cluster_name :
				versions.append(allContent[0])
	return versions

def createClusterDebian (installation) :
	if sys.platform == 'win32' :
		return
	
	cmd = ['usermod','-a','-G','juumpinfinite','postgres']
	shellExecExceptOnError(cmd)

	
	items = [installation._privatekey_file_pg, installation._privatekey_file_pg,installation._ssl_folder_pg]
	for item in items :
		cmd = ['chown','-R','postgres:postgres', item]
		shellExecExceptOnError(cmd)
		
		cmd = ['chmod','-R','400', item]
		shellExecExceptOnError(cmd)
		
	cmd = ['chmod','u+x',installation._ssl_folder_pg]
	shellExecExceptOnError(cmd)
	
	cmd = ['chown','-R','postgres:postgres', installation.postgres_data_folder]
	shellExecExceptOnError(cmd)
		
	cmd = ['chmod','-R','700', installation.postgres_data_folder]
	shellExecExceptOnError(cmd)
	
	if not isClusterExists(installation) :
		cmd = ['pg_createcluster','-d',installation.postgres_data_folder,'--port',str(installation.postgres_port),'-E','UTF8','--locale=en_US.UTF8','-u','postgres','%s' % (installation._postgresl_version),installation._postgresl_cluster_name]
		shellExecExceptOnError(cmd)
		
def configurePostgresConfFile(confFile,pgport,certificate,privateKey) :
	
	src = confFile
	dst = src
	
	bakupFile = os.path.join(os.getcwd(),os.path.split(dst)[1]+".bak")
	if not os.path.exists(bakupFile):
		shutil.copy(dst,bakupFile)
	
	regContent = '^#{0,1}\\s*%s\\s*=.*$'
	
	regs = [
				(re.compile(regContent % 'port'), "port = %s				# (change requires restart)" % pgport),
				(re.compile(regContent % 'ssl'), "ssl = on				  # (change requires restart)"),
				(re.compile(regContent % 'ssl_ciphers'),"ssl_ciphers = 'DEFAULT:!LOW:!EXP:!MD5:@STRENGTH'	 # allowed SSL ciphers # (change requires restart)"),
				#(re.compile(regContent % 'ssl_renegotiation_limit'),"#ssl_renegotiation_limit = 512MB	  # amount of data between renegotiations"),
				(re.compile(regContent % 'ssl_cert_file'),"ssl_cert_file = '%s'		  # (change requires restart)" % certificate),
				(re.compile(regContent % 'ssl_key_file'),"ssl_key_file = '%s'	  # (change requires restart)" % privateKey),
				(re.compile(regContent % 'listen_addresses'),"listen_addresses = 'localhost'	# what IP address(es) to listen on;"),
				(re.compile(regContent % 'shared_buffers'),"shared_buffers = 1GB			# min 128kB"),
				#(re.compile(regContent % 'checkpoint_segments'),"checkpoint_segments = 30		# in logfile segments, min 1, 16MB each"),
				(re.compile(regContent % 'client_min_messages'),"client_min_messages = warning		# values in order of decreasing detail:"),
				(re.compile(regContent % 'logging_collector'),"logging_collector = on		# Enable capturing of stderr and csvlog"),
				(re.compile(regContent % 'max_connections'),"max_connections = 1000			# (change requires restart)"),
				(re.compile(regContent % 'work_mem'),"work_mem = 16MB				# min 64kB"),
				(re.compile(regContent % 'maintenance_work_mem'),"maintenance_work_mem = 512MB		# min 1MB"),
				(re.compile(regContent % 'password_encryption'),"password_encryption = scram-sha-256	# min 1MB"),
			]
			
	handle = open(src,"r")
	oldContent = handle.read()
	handle.close()
	
	allLines = oldContent.split('\n')
	newContent = ''
	lMatchedRe = set()
	for line in allLines :
		if line != '':
			foundContent = False
			for (reg,replacement) in regs :
				if reg.match(line) :
					lMatchedRe.add(reg)
					foundContent = True
					newContent = newContent + replacement + '\n'
					break
			if not foundContent:
				newContent = newContent + line + '\n'
	if len(lMatchedRe) != len(regs) :
		print('Found missmatch %s!=%s, exit' % (len(lMatchedRe),len(regs)))
		sys.exit(1)
		
	if oldContent != newContent :
		handle = open(dst,"w")
		handle.write(newContent)
		handle.close()

def getPreviousPostgresConfFile(installation) :
	if sys.platform == 'win32' :
		return os.path.join(installation._previous_win_postgres_data_folder,'postgresql.conf')	
	else :
		return '/etc/postgresql/%s/%s/postgresql.conf' % (installation._previous_postgres_version,installation._postgresl_cluster_name)
		
def getPostgresPortFromConfFile(confFile) :
	
	reg = re.compile('^\\s*%s\\s*=\\s*([0-9]+)[^0-9]?.*$' % 'port')
				
	handle = open(confFile,"r")
	content = handle.read()
	handle.close()
	
	allLines = content.split('\n')
	found = 0
	port = 5432
	for line in allLines :
		if line != '':
			regObj = reg.match(line)
			if not regObj is None :
				found = found + 1
				port = int(regObj.group(1))
				#print("found port %d" % port)
	if found > 1 :
		raise Exception("Mulitple port definition found for postgres")
	#print("final port %d" % port)
	return port
		
def restoreHbaFile(src,hbaFile) :
	
	dst = hbaFile
	
	bakupFile = os.path.join(os.getcwd(),os.path.split(dst)[1]+".bak")
	if not os.path.exists(bakupFile):
		shutil.copy(dst,bakupFile)
	
	handle = open(src,"r")
	content = handle.read()
	handle.close()
	
	oldContent = ''
	if os.path.exists(dst) :
		handle = open(dst,"r")
		oldContent = handle.read()
		handle.close()
	
	if oldContent != content:
		handle = open(dst,"w")
		handle.write(content)
		handle.close()

def changePostgresPassword(psql,hbaFile,pgport,pgdatabaseowner,pglogin,pgpasswd,serviceName):
		
	os.environ['PGPASSWORD'] = pgpasswd
	pg_cmd = "SELECT * FROM pg_stat_database;"
	cmd = [psql,'-w','-U',pglogin,'-d','postgres','-h','127.0.0.1','-p',str(pgport),'-c',pg_cmd]
	(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	del os.environ['PGPASSWORD'] 

	if errorCode == 0 :
		return
	
	src = hbaFile
	
	(handle,bakupFile) = tempfile.mkstemp(suffix='.tmp')
	os.close(handle)
	shutil.copy(src,bakupFile)
		
	content = 'host all all 127.0.0.1/32 trust'
	handle = open(src,"w")
	handle.write(content)
	handle.close()
	
	restartService(serviceName,altName='postgresql')
	
	if pgdatabaseowner != pglogin :
		pg_cmd = "DO $$ BEGIN IF NOT EXISTS (SELECT FROM pg_catalog.pg_roles WHERE rolname = '%s') THEN CREATE ROLE %s;END IF;END $$;" % (pglogin,pglogin)
		cmd = [psql,'-w','-U',pgdatabaseowner,'-d','postgres','-h','127.0.0.1','-p',str(pgport),'-c',pg_cmd]
		shellExecExceptOnError(cmd)
		pg_cmd =  "ALTER USER %s WITH SUPERUSER CREATEDB LOGIN CREATEROLE INHERIT;" % pglogin
		cmd = [psql,'-w','-U',pgdatabaseowner,'-d','postgres','-h','127.0.0.1','-p',str(pgport),'-c',pg_cmd]
		shellExecExceptOnError(cmd)
	
	pg_cmd = "ALTER USER %s WITH PASSWORD '%s';" % (pglogin,pgpasswd)
	cmd = [psql,'-w','-U',pgdatabaseowner,'-d','postgres','-h','127.0.0.1','-p',str(pgport),'-c',pg_cmd]
	shellExecExceptOnError(cmd)
	
	# and restore original hba
	handle = open(bakupFile,"r")
	content = handle.read()
	handle.close()
	os.remove(bakupFile)
	
	handle = open(src,"w")
	handle.write(content)
	handle.close()
	
	restartService(serviceName,altName='postgresql')

def getPsql (installation):
	if sys.platform == 'win32':
		return os.path.join(installation.postgres_folder,'bin','psql.exe')
	else :
		return 'psql'

def getPreviousPsql (installation):
	if sys.platform == 'win32':
		return os.path.join(installation._previous_win_postgres_folder,'bin','psql.exe')
	else :
		return 'psql'

def getPreviousPostgresqlServiceName(installation) :
	if sys.platform == 'win32':
		return 'postgresql-x64-%s'%(installation._previous_postgres_version)
		
	else :
		return 'postgresql'

def getPostgresqlServiceName(installation) :
	if sys.platform == 'win32':
		return 'postgresql-x64-%s'%(installation._postgresl_version)
		
	else :
		return 'postgresql'

def getPostgresqlHbaFile(installation) :
	if sys.platform == 'win32':
		return os.path.join(installation.postgres_data_folder,'pg_hba.conf')
	else :
		return '/etc/postgresql/%s/%s/pg_hba.conf' % (installation._postgresl_version,installation._postgresl_cluster_name)

def getPreviousPostgresqlHbaFile(installation) :
	if sys.platform == 'win32':
		return os.path.join(installation._previous_win_postgres_data_folder,'pg_hba.conf')
	else :
		return '/etc/postgresql/%s/%s/pg_hba.conf' % (installation._previous_postgres_version,installation._postgresl_cluster_name)


def checkHasBuilds(installation) :
	
	print("Checking if previous installation has some builds")
	
	src = getPreviousPostgresqlHbaFile(installation)
	if not os.path.exists(src) :
		return
	
	old_port = getPostgresPortFromConfFile(getPreviousPostgresConfFile(installation))
	
	(handle,bakupFile) = tempfile.mkstemp(suffix='.tmp')
	os.close(handle)
	shutil.copy(src,bakupFile)

	content = 'host all all 127.0.0.1/32 trust'
	handle = open(src,"w")
	handle.write(content)
	handle.close()
	
	restartService(getPreviousPostgresqlServiceName(installation),altName='postgresql')
	
	psql = getPreviousPsql(installation)

	hasBuildPattern = '####3djuump-infinite Has Builds####'
	hasNoBuildPattern = '####3djuump-infinite Has No Builds####'
	
	build_version = os.environ['SV_BUILD_VERSION']
	build_versions = build_version.split('.')
	
	pg_cmd	=	"DROP TABLE IF EXISTS infinite_temp; "
	pg_cmd +=	"DROP FUNCTION IF EXISTS testHasBuilds(); "
	pg_cmd +=	"CREATE OR REPLACE FUNCTION testHasBuilds() "
	pg_cmd +=	"RETURNS TABLE(build_content TEXT) "
	pg_cmd +=	"LANGUAGE plpgsql "
	pg_cmd +=	"AS $$ "
	pg_cmd +=	"DECLARE "
	pg_cmd +=	"BEGIN "
	pg_cmd +=		"build_content = '%s'; " % hasNoBuildPattern
	pg_cmd +=		"IF EXISTS (SELECT FROM information_schema.tables WHERE table_schema='DirectoryPrivate' AND table_name='ProxyBuilds') THEN "
	pg_cmd +=			"IF EXISTS ( SELECT FROM \"DirectoryPrivate\".\"ProxyBuilds\" WHERE string_to_array(buildversion, '.')::int[] != ARRAY[%s,%s]) THEN " % (build_versions[0],build_versions[1]) 
	pg_cmd +=				"build_content = '%s'; " % hasBuildPattern
	pg_cmd +=			"END IF; "
	pg_cmd +=		"END IF;	 "
	pg_cmd +=		"IF EXISTS (SELECT FROM information_schema.tables WHERE table_schema='Directory' AND table_name='ProxyBuilds') THEN "
	pg_cmd +=			"IF EXISTS (SELECT FROM \"Directory\".\"ProxyBuilds\" WHERE string_to_array(builddbcommentlight->>'buildversion', '.')::int[] != ARRAY[%s,%s]) THEN " % (build_versions[0],build_versions[1])
	pg_cmd +=				"build_content = '%s'; " % hasBuildPattern
	pg_cmd +=			"END IF; "
	pg_cmd +=		"END IF; "
	pg_cmd +=		"IF (build_content = '####3djuump-infinite Has No Builds####') THEN "
	pg_cmd +=			"IF EXISTS (SELECT FROM information_schema.tables WHERE table_schema='ProxyAdministration' AND table_name='Builds') THEN "
	pg_cmd +=				"IF EXISTS (SELECT FROM information_schema.columns WHERE table_name='Builds' and column_name='buildproperties') THEN "
	pg_cmd +=					"IF EXISTS (SELECT FROM \"ProxyAdministration\".\"Builds\" WHERE string_to_array(buildproperties->>'buildversion', '.')::int[] != ARRAY[%s,%s]) THEN " % (build_versions[0],build_versions[1])
	pg_cmd +=						"build_content = '%s'; " % hasBuildPattern
	pg_cmd +=					"END IF; "
	pg_cmd +=				"ELSE "
	pg_cmd +=					"IF EXISTS (SELECT dbnameorbuildid FROM \"ProxyAdministration\".\"Builds\") THEN "
	pg_cmd +=						"build_content = '%s'; " % hasBuildPattern
	pg_cmd +=					"END IF; "
	pg_cmd +=				"END IF; "
	pg_cmd +=			"END IF; "
	pg_cmd +=		"END IF; "
	pg_cmd +=		"RETURN NEXT; "
	pg_cmd +=	"END; "
	pg_cmd +=	"$$; "

	pg_cmd +=	"CREATE TEMPORARY TABLE infinite_temp AS (SELECT * FROM testHasBuilds()); "
	pg_cmd +=	"DROP FUNCTION IF EXISTS testHasBuilds(); "
	pg_cmd +=	"SELECT * FROM infinite_temp; "
	
	databases = ['InfiniteDirectory', 'InfiniteProxy']
	
	lHasErrors = False
	for cur_database in databases :
		cmd = [psql,'-w','-U',installation.postgres_login,'-d',cur_database,'-h','127.0.0.1','-p','%d' % old_port,'-c',pg_cmd]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
	
		#print(stdout)
		#print('####')
		#print(stderr)
		
		if errorCode == 0:
			if stdout.find(hasBuildPattern) >= 0 :
				lHasErrors = True
	
	# and restore original hba
	handle = open(bakupFile,"r")
	content = handle.read()
	handle.close()
	os.remove(bakupFile)
	
	handle = open(src,"w")
	handle.write(content)
	handle.close()
	
	restartService(getPreviousPostgresqlServiceName(installation),altName='postgresql')
	
	if lHasErrors:
		print("The 3djuump infinite cluster has still some builds, please remove all builds (and backup as evojuumps) and projets (and backup their project descriptors) before proceeding")
		sys.exit(1)
		
## configure postgres : hba.conf and postgresql.conf
def configurePostgresImpl(installation) :
	
	createClusterDebian(installation)
	
	print("Configuring postgresql")
	
	psql = getPsql(installation)
	serviceName = getPostgresqlServiceName(installation)
	hbaFile = getPostgresqlHbaFile(installation)
	if sys.platform == 'win32':
		confFile = os.path.join(installation.postgres_data_folder,'postgresql.conf')	
		pgdatabaseowner = installation.postgres_login
		certificate = installation.certificate_file.replace('\\','/')
		privateKey = installation.privatekey_file.replace('\\','/')
	else :
		confFile = '/etc/postgresql/%s/%s/postgresql.conf' % (installation._postgresl_version,installation._postgresl_cluster_name)
		pgdatabaseowner = 'postgres'
		certificate = installation._certificate_file_pg
		privateKey = installation._privatekey_file_pg
	
	# configure and set the required port and certificate
	configurePostgresConfFile(confFile,installation.postgres_port,certificate,privateKey)
	
	# restore hba
	restoreHbaFile(os.path.join(installation._Djuump_installers,'install form','config','pg_hba.conf'),hbaFile)
	
	# restart service to reflect new conf
	restartService(serviceName,altName='postgresql')	
	
	# reset pwd
	changePostgresPassword(psql,hbaFile,installation.postgres_port,pgdatabaseowner, installation.postgres_login,installation.postgres_password,serviceName)
	
# executes a program but allows user input
def shellExecReturn (cmd):
	# make sure we have a list
	if type(cmd) is str :
		import shlex
		cmd = shlex.split(cmd)
	
	errorCode = 0
	try :
		errorCode = subprocess.call(cmd)
	except :
		print("Error executing %s "% cmd[0])
		errorCode = 1
		
	return errorCode

def stopService(serviceName) :
	if sys.platform == 'win32':
		cmd = ['net','stop',serviceName,'/y']
	else :
		cmd = ['service',serviceName,'stop']
	
	# ignore error, service might be missing
	shellExecExceptOnError(cmd,True)

# restart the given service
def restartService(serviceName, pIgnoreError = False, altName=None) :
	
	printName = serviceName
	if not altName is None :
		printName = altName
	print('restarting service %s'% printName)
	if sys.platform == 'win32':
		stopService(serviceName)
		cmd = ['net','start',serviceName]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		if errorCode != 0 and not pIgnoreError:
			raise Exception('Error restarting %s, return %d %s %s' % (printName,errorCode,stdout,stderr))
	else :
		
		cmd = ['service',serviceName,'restart']
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		if errorCode != 0 and not pIgnoreError:
			raise Exception('Error restarting %s, return %d\nstdout:\n%s\nstderr\n%s' % (printName,errorCode,stdout,stderr))
			
# restart the given service
def uninstallService(serviceName) :
	if sys.platform == 'win32':
		cmd = ['sc','delete',serviceName]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		if errorCode != 0 :
			raise Exception('Error unininstalling %s, return %d %s %s' % (serviceName,errorCode,stdout,stderr))

def stopAllServices(installation) :
	stopService('3djuump-infinite-server')
	stopService('3djuump-infinite-proxy')
	stopService('3djuump-infinite-directory')
	stopService(getPostgresqlServiceName(installation))
	if sys.platform != 'win32':
		stopService('apache2')
		stopService('elasticsearch')
	else:
		stopService('Apachehttp3djuumpInfinite')
		stopService('elasticsearch-service-x64')
		
def toCsv (src,dst,sheetname,curversion,separator,values = dict()) :
	import openpyxl,sys,os
	
	wb2 = openpyxl.load_workbook(src)
	worksheet = wb2[sheetname]
	worksheet['C2'].value = curversion
	for key,item in values.items() :
		worksheet[key].value = item
	
	sys.stderr = open(os.devnull, "w")
	highRow = worksheet.get_highest_row()
	highColumn = worksheet.get_highest_column()
	sys.stderr = sys.__stdout__
	
	content = ''
	lastEmptyLine = True
	for curRow in range(1,highRow+1):
		lineValue = ''
		contentCounter = 0
		
		for curColumn in range(1,highColumn+1):
			cellContent = worksheet.cell(row = curRow, column = curColumn).value
			if cellContent == None :
				lineValue = lineValue+separator
			else :
				contentCounter = contentCounter + 1
				lineValue = lineValue+str(cellContent) + separator
		if contentCounter > 1 :
			content = content + lineValue + "\n"
			lastEmptyLine = False
		else :
			if not lastEmptyLine:
				content = content + "\n"
			lastEmptyLine = True
	handle = open(dst,"w")
	handle.write(content)
	handle.close()

def getCurrentAttribute(item,attrname) :
	if hasattr(item,attrname) :
		return getattr(item,attrname)
	return ''
		
def getAttributes (item) :
	allNames = dir(item)
	allAttr = []
	for name in allNames :
		if not name.startswith('_') and not callable(getattr(item, name)):
			allAttr.append(name)
	return allAttr

def printAttrs (item) :
	allAttr = getAttributes(item)
	for name in allAttr :
		print(name+" : "+str(getattr(item, name)))


def toUrl (inputPath) :	
	if sys.version_info[0] < 3:		
		import urlparse, urllib
		return urlparse.urljoin(
			'file:', urllib.pathname2url(inputPath))
	else :
		import pathlib
		return pathlib.Path(inputPath).as_uri()

def getAppData () :
	# import ctypes
	# from ctypes import wintypes, windll

	# CSIDL_COMMON_APPDATA = 35
	# _SHGetFolderPath = windll.shell32.SHGetFolderPathW
	# _SHGetFolderPath.argtypes = [wintypes.HWND,
	# ctypes.c_int,
	# wintypes.HANDLE,
	# wintypes.DWORD, wintypes.LPCWSTR]

	# path_buf = wintypes.create_unicode_buffer(wintypes.MAX_PATH)
	# result = _SHGetFolderPath(0, CSIDL_COMMON_APPDATA, 0, 0, path_buf)
	# print(path_buf.value)
	# return path_buf.value
	if _haspywin32 :
		from win32com.shell import shellcon, shell
		homedir = shell.SHGetFolderPath(0, shellcon.CSIDL_COMMON_APPDATA, 0, 0)
		
	else :
		import ctypes
		
		CSIDL_COMMON_APPDATA = 35
		_SHGetFolderPath = ctypes.windll.shell32.SHGetFolderPathW
		_SHGetFolderPath.argtypes = [ctypes.wintypes.HWND,
							ctypes.c_int,
							ctypes.wintypes.HANDLE,
							ctypes.wintypes.DWORD, ctypes.wintypes.LPCWSTR]

		path_buf = ctypes.create_unicode_buffer(ctypes.wintypes.MAX_PATH)
		result = _SHGetFolderPath(0, CSIDL_COMMON_APPDATA, 0, 0, path_buf)
		homedir = path_buf.value
	return homedir


def loadJson(src) :
	with open(src,'r',encoding='utf-8') as f:
		return json.load(f)
	raise Exception('fail to read %s' % (src))
	
def saveJson(input,dst):
	with open(dst,'w',encoding='utf-8') as f:
		return json.dump(input,f, sort_keys=True, indent=4)
	raise Exception('fail to write %s' % (dst))

def parseIni(src) :
	parser = MyParser()
	parser.read(src)
	d = parser.as_dict()
	return d

def saveIni(input,dst) :
	content = ''
	
	if 'General' in input :
		content = content + ('[General]\n')
		for item,itemVal in input['General'].items() :
			content = content + ('%s=%s\n' % (item,itemVal))
		content = content+"\n"
		input.pop("General", None)
			
	for key,val in input.items() :
		content = content + ('[%s]\n' % key)
		for item,itemVal in val.items() :
			content = content + ('%s=%s\n' % (item,itemVal))
		content = content+"\n"
		
	handle = open(dst,"w")
	handle.write(content)
	handle.close()

def saveConf (input,packageName,dst) :
	content = ''
			
	for key,val in input.items() :
		content = content + ('%s\t%s\t%s\t%s\n' % (packageName, key,val[0],val[1]))
		
	handle = open(dst,"w")
	handle.write(content)
	handle.close()
	
def copyFiles(inputFolder,outFolder) :
	files = glob.glob(os.path.join(inputFolder,'*'))
	for curFile in files :
		dst = os.path.join(outFolder,os.path.split(curFile)[1])
		if not os.path.exists(dst) :
			shutil.copy(curFile,dst)

def installInfiniteBinary(installation,exename,serviceName, values, unholdPackages, checkApt = True) :
	
	if sys.platform == 'win32' :
		print("Installing %s" % serviceName)
		
		filepattern = os.path.join(installation._Djuump_installers,'dist','win',exename) + "*"
		allFiles = glob.glob(filepattern)
		if len(allFiles) != 1:
			raise Exception("Could not find file %s" % exename)
			
		(handle,tmp_file) = tempfile.mkstemp(suffix='.ini')
		os.close(handle)
		saveIni(values,tmp_file)
		
		cmd = [os.path.normpath(allFiles[0]),'/SILENT','/LOADINF=%s' % tmp_file]
		
		cmd = cmd + ['/LOG=./' + exename + '.log']
	
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		os.remove(tmp_file)
		if errorCode != 0 :
			raise Exception("Error installing %s %d %s %s" % (cmdToString(cmd),errorCode,stdout,stderr))
		
		
	else:
		for item in unholdPackages :
			cmd = cmd = ['apt-mark','unhold', item]
			shellExecExceptOnError(cmd)
		uninstallPackage(exename)
		
		if installation._first_linux_binary_install :
			installation._first_linux_binary_install = False
			uninstallPackage("lib3djuump-infinite-common")
		
		(handle,tmp_file) = tempfile.mkstemp(suffix='.conf')
		os.close(handle)
		saveConf(values,serviceName, tmp_file)
		cmd = ['debconf-set-selections',tmp_file]
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		os.remove(tmp_file)
		if errorCode != 0 :
			raise Exception("Error registering selections")
		installPackage(exename,False,True)
		
		
		if checkApt and installation._run_apt_upgrade :
			installation._run_apt_upgrade = False
			lastfrontend = None
			if 'DEBIAN_FRONTEND' in os.environ:
				lastfrontend=os.environ['DEBIAN_FRONTEND']
			os.environ['DEBIAN_FRONTEND'] = 'noninteractive'
			print('Running apt-get upgrade')
			cmd = ['apt-get','-yq','-o','Dpkg::Options::=--force-confdef','upgrade']
			shellExecExceptOnError(cmd)
			if lastfrontend is None :
				del os.environ['DEBIAN_FRONTEND']
			else :
				os.environ['DEBIAN_FRONTEND']= lastfrontend


class commonInstallationInfos :


	def __init__(self) :
		
		self._postgresl_cluster_name = 'main'
		self._postgresl_installer = 'postgresql-' + os.environ['SV_VERSION_PG_FULL'] + '-1-windows-x64.exe'
		self._postgresl_version = os.environ['SV_VERSION_PG_SMALL']
		self._plugin_installer = '3DJuump Infinite Postgres Plugins x64-setup-'
		self._apache_plugin_installer = '3DJuump Infinite Apache Plugins x64-setup-'
		self._java_installer = 'openjdk1.8.0_' + os.environ['SV_VERSION_JRE'] + '-windows-x64.zip'
		self._elasticsearch_installer = 'elasticsearch-' + os.environ['SV_VERSION_ES_FULL'] + '.zip'
		self._eseditor_installer = '3djuumpInfiniteEsEditorHtml_*.zip'
		self._redists = [('vcredist_x64_v140','/quiet'),('vcredist_x64_v120','/quiet'),('vcredist_x64_v110','/quiet'),('vcredist_x64_v100','/quiet')]
		self._from_basepath = False
	
		# where install folder is located
		self._Djuump_installers = ''
		self._install_apache = True
		self._install_postgres = True
		
		self._install_elastic = True
		self._run_apt_upgrade = False
		self._ssl_folder_pg = ''
		self._certificate_file_pg = ''
		self._privatekey_file_pg = ''
		
		self._previous_win_postgres_folder = ''
		self._previous_postgres_version = ''
		self._previous_win_postgres_data_folder = ''
		self._first_linux_binary_install = True
		
		self._httpd_conf_custom_includes = []
		self._httpd_infinite_conf_custom_includes = ''
		
		if sys.platform != "win32" :
			self.getLinuxDistribution()
		self._Djuump_installers = os.path.normpath(os.path.abspath('..')).replace('\\','/')
	
		print('Options :\n\t--accept-eula : to skip eula consent\n\t-f to force other consents')
	
	def getLinuxDistribution (self) :
		cmd = ['lsb_release','-a']
		(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
		reg =r'.*Codename:\s*([^\s]+).*'
		reg = re.compile(reg)
		result = reg.search(stdout)
		if result is None:
			print("Cannot get distribution codename, is lsb-release installed ?")
			print("exiting")
			sys.exit(1)
		self.linux_distribution = result.group(1)

	## creates the ssl key certificate pair
	def createSSL (self) :
		createSSLFiles(self)
		
		## installs postgres silently
	def installPostgres(self) :
		installPostgresImpl(self)
		
	def installApache(self):
		installApacheImpl(self)
	
	def installPostgresPlugins(self):
		installPostgresPluginsImpl(self)
	
	def installApachePlugins(self):
		installApachePluginsWin(self)
	
	def stopInfiniteServices(self):
		stopAllServices(self)
	
	def installElasticSearch(self):
		if sys.platform == 'win32' :
			if not self._install_elastic :
				print("Skipping elastic search installation")
			else :
				# remove old version if required
				(javafolder,elasticfolder) = getElasticSearchFolders()
				if elasticfolder != '':
					print("Stoping ElasticSearch service")
					stopService('elasticsearch-service-x64')
					time.sleep(2)
					print("Removing ElasticSearch")
					uninstallService('elasticsearch-service-x64')
					time.sleep(2)
					goon = True
					while goon :
						try :
							if os.path.exists(elasticfolder):
								shutil.rmtree(elasticfolder)
							goon = False
						except:
							time.sleep(1)
					
				if javafolder != '':
					print("Removing Java")
					shutil.rmtree(javafolder)
			
				print("Extracting java")
				zfile = zipfile.ZipFile(os.path.join(self._Djuump_installers,'third-party',self._java_installer))
				zfile.extractall(self.java_binary_folder)
				
				print("Extracting elasticsearch")
				zfile = zipfile.ZipFile(os.path.join(self._Djuump_installers,'third-party',self._elasticsearch_installer))
				for zinfo in zfile.infolist():
					dst = self.elastic_binary_folder + zinfo.filename.replace('elasticsearch-' + os.environ['SV_VERSION_ES_FULL'],'',1)
					if zinfo.filename.endswith('/'):
						os.makedirs(dst)
					else :
						dstFolder = os.path.split(dst)[0]
						if not os.path.exists(dstFolder) :
							os.makedirs(dstFolder)
						content = zfile.read(zinfo)
						handle = open(dst,"wb")
						handle.write(content)
						handle.close()
				zfile.close()
		else :
			if self.linux_oracle_java != True:
				java_package = 'default-jre'
				# upgrade if necessary
				installPackage(java_package)
			# upgrade if necessary
			installPackage('elasticsearch')
			# register to launch on startup
			cmd=['update-rc.d','elasticsearch','defaults','95','10']
			shellExecExceptOnError(cmd,True)
			cmd=['systemctl','enable','elasticsearch']
			shellExecExceptOnError(cmd,True)
		
		# patch jvm.options here, before registering the service
		if sys.platform == 'win32' :
			jvmoptfile = os.path.join(self.elastic_binary_folder,'config','jvm.options')
		else :
			jvmoptfile = os.path.join('/etc/elasticsearch/jvm.options')
		bakupFile = os.path.join(os.getcwd(),os.path.split(jvmoptfile)[1]+".bak")
		if not os.path.exists(bakupFile):
			shutil.copy(jvmoptfile,bakupFile)
		
		newcontent = ''
		handle = open(jvmoptfile,"r")
		content = handle.read()
		handle.close()
		lOriginalContent = content
		
		
		content = content.replace('-Xms1g','-Xms4g')
		content = content.replace('-Xmx1g','-Xmx4g')
		
		content = content.replace('8:-XX:+PrintGCDetails','8:-XX:-PrintGCDetails')
		content = content.replace('8:-XX:+PrintGCApplicationStoppedTime','8:-XX:-PrintGCApplicationStoppedTime')
		
		if content != lOriginalContent:
			handle = open(jvmoptfile,"w")
			handle.write(content)
			handle.close()
			
	def patchElasticSearch(self):
		print("patching elasticsearch")
		
		if sys.platform == 'win32':
			src = os.path.join(self.elastic_binary_folder,'bin','elasticsearch-service.bat')
			dst = os.path.join(self.elastic_binary_folder,'bin','service.bat')
			
			java_home_folder = os.path.join(self.java_binary_folder,'jdk1.8.0_' + os.environ['SV_VERSION_JRE']);
			
			content = '@echo off\n'
			content = content + 'set JAVA_HOME=' + java_home_folder + '\n'
			handle = open(src,"r")
			content = content + handle.read()
			handle.close()
			handle = open(dst,"w")
			handle.write(content)
			handle.close()
			
			
			stopService('elasticsearch-service-x64')
			
			cmd = ['sc','delete','elasticsearch-service-x64','/Q']
			shellExecExceptOnError(cmd,True)
			
			cmd = ['sc','query','elasticsearch-service-x64']
			while True:
				(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
				if errorCode != 0:
					break;
				time.sleep(1)
				
			
			print('Install elasticsearch-service-x64')
			cmd = [os.path.join(self.elastic_binary_folder,'bin','service.bat'),'install']
			shellExecExceptOnError(cmd)
			
			# service.bat keep %JAVA_HOME% in settings 
			# we don't want to set it system wide, so we tweak reg entry were it is stored to replace it
			if sys.hexversion < 0x03000000:
				import _winreg as reg
			else :
				import winreg as reg
			
			lKeyName = r'SOFTWARE\Wow6432Node\Apache Software Foundation\Procrun 2.0\elasticsearch-service-x64\Parameters\Java\\'

			lRegKey = reg.OpenKey(reg.HKEY_LOCAL_MACHINE,lKeyName,0,reg.KEY_WRITE)
			lJvmDllPath = os.path.join(java_home_folder,r'bin\server\jvm.dll')
			reg.SetValueEx(lRegKey,'Jvm',0,reg.REG_SZ,lJvmDllPath)
			reg.CloseKey(lRegKey)
			
			cmd = ['sc','config','elasticsearch-service-x64','start=','auto' ]
			shellExecExceptOnError(cmd)
			
	def configureElasticSearch(self):
		try:
			import urllib.request as urllib2
		except:
			import urllib2
			
		print("Configuring elasticsearch")
		
		restart_service = True
		
		if sys.platform == 'win32' :
			src = os.path.join(self.elastic_binary_folder,'config','log4j2.properties')
			serviceName = 'elasticsearch-service-x64'
		else :
			cmd= ['usermod','-a','-G','juumpinfinite','elasticsearch']
			shellExecExceptOnError(cmd)
			
			src = os.path.join('/etc/elasticsearch/log4j2.properties')
			serviceName = 'elasticsearch'
			
		dst = src
		
		bakupFile = os.path.join(os.getcwd(),os.path.split(dst)[1]+".bak")
		if not os.path.exists(bakupFile):
			shutil.copy(dst,bakupFile)
		
		handle = open(src,"r")
		content = handle.read()
		handle.close()
		
		regs = []
		regs.append((re.compile('^\\s*rootLogger\\.level\\s*=.*'),'rootLogger.level = error'))
		regs.append((re.compile('^\\s*logger\\.action\\.level\\s*=.*'),'logger.action.level = info'))
		
		allLines = content.splitlines()
		
		newContent = ''
		count = 0
		for line in allLines :
			if line != '' :
				foundContent = False
				for (reg,replacement) in regs :
					matchObj = reg.match(line)
					if matchObj :
						count = count + 1
						foundContent = True
						newContent = newContent + replacement + '\n'
						break
				if not foundContent:
					newContent = newContent + line + '\n'
					
		if count != len(regs):
			raise Exception('error configuring elastic search logging')
			
		if newContent != content :
			handle = open(dst,"w")
			handle.write(newContent)
			handle.close()
		
		src = os.path.join(self._Djuump_installers,'install form','config','elasticsearch.yml')
		
		if sys.platform == 'win32' :
			dst = os.path.join(self.elastic_binary_folder,'config','elasticsearch.yml')
		else :
			dst = '/etc/elasticsearch/elasticsearch.yml'
			
		bakupFile = os.path.join(os.getcwd(),os.path.split(dst)[1]+".bak")
		
		if not os.path.exists(bakupFile):
			shutil.copy(dst,bakupFile)
			
		copyAndConfigure(self,src,dst,[])
		
		# if sys.platform != 'win32' :
			# if configureElasticDebian(self):
				# restart_service = True
		
		# if sys.platform != 'win32' :
			# if not isProcessRunning('java') :
				# restart_service = True
		
		if restart_service :
			# don't now why sometimes restart will fail ...
			for i in range(5,-1,-1):
				try:
					restartService(serviceName, pIgnoreError = (i > 0),altName='elasticsearch' )
					break;
				except:
					if i == 0:
						raise Exception('Fail to restart elasticsearch, abort')
					else:
						print('Failed to restart elasticsearch, will retry in a few moment')
						time.sleep(10.)
						
			# wait for elastic search service
			if 'http_proxy' in os.environ :
				http_proxy=os.environ['http_proxy']
				del os.environ['http_proxy']
			else:
				http_proxy=None
			
			if 'https_proxy' in os.environ :
				https_proxy=os.environ['https_proxy']
				del os.environ['https_proxy']
			else:
				https_proxy=None

			for i in range(10,-1,-1):
				lUrl = 'http://127.0.0.1:' + str(self.elastic_port)
				
				request = urllib2.Request(lUrl)
				lResponse = None
				try:
					lHttpHandler = urllib2.HTTPHandler()#debuglevel=2
					lProxyHandler = urllib2.ProxyHandler({})
					lOpener = urllib2.build_opener(lProxyHandler,lHttpHandler)
					lResponse = lOpener.open(request)
				except:
					lResponse = None
				if lResponse is None or lResponse.getcode() != 200:
					if i == 0:
						raise Exception('elasticsearch is not responding, abort')
					else:
						print('elasticsearch is not responding, will retry in a few moment')
						time.sleep(10.)
			if not http_proxy is None :
				os.environ['http_proxy']=http_proxy
			
			if not https_proxy is None :
				os.environ['https_proxy']=https_proxy
			print('elasticsearch is ready')
	
	def patchApacheProxy(self):
		self.patchApache('3djuumpproxy_empty')
	
	def patchApacheDirectory(self):
		self.patchApache('3djuumpdirectory_empty')
		
	def patchApache(self, configFile):
		ports = ['127.0.0.1:%s'%self.apache_https_port]
		
		if sys.platform == 'win32' :
		
			print("Patching apache")
			
			cmd = ['icacls',os.path.normpath(self.apache_folder),'/grant','*S-1-5-20:(OI)(CI)F','/t','/q']
			shellExecExceptOnError(cmd)
			
			cmd= ['sc','config','Apachehttp3djuumpInfinite','obj=','NT AUTHORITY\\NetworkService']
			shellExecExceptOnError(cmd)
			
			cmd = ['sc','config','Apachehttp3djuumpInfinite','start=','auto' ]
			shellExecExceptOnError(cmd)
			
		configureApacheImpl(self,[],ports,configFile,False)
	
	## configure postgres : hba.conf and postgresql.conf
	def configurePostgres(self) :
		configurePostgresImpl(self)
	
	def checkPrerequisites(self):
		checkAdmin()
		
		if sys.platform != 'win32':
			if not '/usr/sbin' in os.environ['PATH'] or not '/sbin' in os.environ['PATH']:
				raise Exception('"/usr/sbin" or "/sbin" are not available in $PATH (use "su -" instead of "su", or ALWAYS_SET_PATH yes in /etc/login.defs)')
		
			checkAptVersion(self)
	
	def prepareSystem(self):
		if sys.platform == 'win32' :
			self.installRedists()
		else :
			configureDebianSystem(self)
		
	def installRedists(self):
		print("Installing redistributables")
		for (exename,option) in self._redists :
			cmd = [os.path.normpath(os.path.join(self._Djuump_installers,'third-party','vcresdists',exename)),option]
			(stdout,stderr,errorCode) = shellExecExceptOnError(cmd,True)
			if errorCode != 0 :
				print("Error installing %s" % exename)
	
	def configureEsEditor(self) :
		src = os.path.join(self._Djuump_installers,'dist','share')
		dstRefFolder = os.path.join(self.proxy_eseditor_folder,'public')
		
		allFiles = glob.glob(os.path.join(src,self._eseditor_installer))
		if len(allFiles) != 1:
			raise Exception("Could not find file %s" % self._eseditor_installer)
			
		zipFilename = allFiles[0]
		
		print("Extracting eseditor")
		zfile = zipfile.ZipFile(zipFilename)
		for zinfo in zfile.infolist():
			if zinfo.filename.endswith('/'):
				continue
			dst = os.path.join(dstRefFolder,zinfo.filename)
			dstFolder = os.path.split(dst)[0]
			if not os.path.exists(dstFolder) :
				os.makedirs(dstFolder)
			content = zfile.read(zinfo)
			handle = open(dst,"wb")
			handle.write(content)
			handle.close()
		zfile.close()
	
	def fromYaml(item, src,srcschema):
		import urllib
		try:
			import yaml
			import jsonschema
			import requests
		except:
			raise Exception('need following modules : PyYaml jsonschema requests')
			
		if not os.path.exists(src) :
			raise Exception("File %s does not exist, exiting" % src)
		
		# load configuration
		lConf = None
		with open(src,encoding='utf8') as f:
			lConf = yaml.safe_load(f)
		
		# convert into json (convert all keys into string)
		lConf = json.loads(json.dumps(lConf))
		
		# load validation schema
		lSchema = None
		with open(srcschema,encoding='utf8') as f:
			lSchema = yaml.safe_load(f)
		
		# validate configuration
		try:
			jsonschema.validate(instance=lConf,schema=lSchema)
		except jsonschema.exceptions.ValidationError as err:
			print('in section : ''%s''' % ('/'.join(err.path)))
			print('\t %s' % (err.message))
			lConf = None
		except:
			raise
		if lConf is None:
			raise Exception('Invalid install configuration')
			
		# parse configuration
		for section in lConf:
			lSubConf = lConf[section]
			if section == 'openidconnect':
				# keep this section in a subobject to avoid key conflicts
				item.openidconnect = lSubConf
				continue
			for key in lSubConf:
				lValue = lSubConf[key]
				if lValue is None:
					continue
				# if field is a file name, replace it with absolute path because installers 
				# might use a different execution folder
				if isinstance(lValue, str) and (os.path.isfile(lValue) or os.path.isdir(lValue)):
					lValue = os.path.abspath(lValue)
				if hasattr(item,key):
					raise Exception('Got attribute conflict while parsing section %s : %s' % (section,key))
				setattr(item, key, lValue)

		# now handle hostname
		if (not hasattr(item,'hostname')) or (item.hostname == '') :
			import socket
			item.hostname = socket.getfqdn()
			if item.hostname == '' or item.hostname == 'localhost':
				item.hostname = socket.gethostname()
				if item.hostname == '':
					item.hostname = 'infinite.local'
		
		# compute publichostname
		if hasattr(item,'proxy_public_url'):
			lProxyPublicUrl = urllib.parse.urlparse(item.proxy_public_url)
			item.publichostname = lProxyPublicUrl.hostname
		else:
			lDirectoryPublicUrl = urllib.parse.urlparse(item.directory_public_url)
			item.publichostname = lDirectoryPublicUrl.hostname
			

		# check for empty attributes
		allItems = getAttributes(item)
		for curAttr in allItems :
			if str(getattr(item, curAttr)) == '':
				should_exit = True
				if should_exit:
					print("attribute %s cannot be empty, exiting" % curAttr)
					sys.exit(1)
		
		# apply proxy to current env for http calls made from this script
		if hasattr(item,'http_proxy') and item.http_proxy != None:
			os.environ['http_proxy'] = item.http_proxy
		if hasattr(item,'https_proxy') and item.https_proxy != None:
			os.environ['https_proxy'] = item.https_proxy
		
		print('http_proxy : %s' % (os.environ['http_proxy'] if 'http_proxy' in os.environ else 'n/a'))
		print('https_proxy : %s' % (os.environ['https_proxy'] if 'https_proxy' in os.environ else 'n/a'))
		
		# some system specific checks
		if sys.platform == 'win32' :
			if getattr(item,'binary_basepath') is None or str(getattr(item,'binary_basepath')) == '':
				raise Exception(' ''binary_basepath'' could not be empty')
		else:
			if getattr(item,'realfusio_apt') is None or str(getattr(item,'realfusio_apt')) == '':
				raise Exception(' ''realfusio_apt'' could not be empty')
			if hasattr(item,'binary_basepath') and (item.binary_basepath != '' or item.binary_basepath != None):
				raise Exception(' ''binary_basepath'' should be empty')
			setattr(item,'binary_basepath','/')
		
		
		# analyze public urls 
		# and find out what we are installing
		lPublicUrl = None
		lIsInstallDirectory = False
		lIsInstallProxy = False
		if not 'proxy' in lConf:
			# we are installing a directory only (remove /directory)
			lPublicUrl = urllib.parse.urlparse(item.directory_public_url[:-10])
			lIsInstallDirectory = True
		elif not 'directory_api_local_port' in lConf['directory']:
			# we are installing a proxy/server only (remove /proxy)
			lPublicUrl = urllib.parse.urlparse(item.proxy_public_url[:-6])
			lIsInstallProxy = True
		else:
			# we are installing a directory + proxy/server (remove /proxy and /directory)
			lPublicUrl = urllib.parse.urlparse(item.directory_public_url[:-10])
			lOtherUrl = urllib.parse.urlparse(item.proxy_public_url[:-6])
			if lPublicUrl.path != lOtherUrl.path:
				raise Exception('inconsistent directory and proxy url')
			if lPublicUrl.port != lOtherUrl.port:
				raise Exception('inconsistent directory and proxy port')
			
			item.directory_postgres_login = item.postgres_login
			item.directory_postgres_password = item.postgres_password
			
			lIsInstallDirectory = True
			lIsInstallProxy = True

		# generate apache configuration based on public url
		setattr(item,'apache_https_port',lPublicUrl.port)
		setattr(item,'apache_path_prefix',lPublicUrl.path)
		if item.apache_path_prefix == '/':
			item.apache_path_prefix=''
		item.apache_path_prefix_regexp = item.apache_path_prefix.replace('/','\\/')

		# define attributes based on what service we are installing
		if lIsInstallDirectory:
			item.directory_service_name = '3djuump-infinite-directory'
			if sys.platform == 'win32' :
				item.directory_exe_name = '3dJuumpInfiniteDirectoryService.exe'
				item.directory_binary_installer = '3DJuump Infinite Directory x64-setup-'
			else:
				item.directory_exe_name = '3dJuumpInfiniteDirectoryService'
				item.directory_binary_installer = '3djuump-infinite-directory'
		
		lDefaultProxyBinaryFolder = None
		item.apache_conf_file_name = ''
		if lIsInstallProxy:
			if item.proxy_role == 'server':
				item.proxy_service_name = '3djuump-infinite-server'
				lDefaultProxyBinaryFolder = '3DJuumpInfiniteServerx64'
				item.apache_conf_file_name = 'server'
				if sys.platform == 'win32' :
					item.proxy_exe_name = '3dJuumpInfiniteServerService.exe'
					item.proxy_binary_installer = '3DJuump Infinite Server x64-setup-'
				else:
					item.proxy_exe_name = '3dJuumpInfiniteServerService'
					item.proxy_binary_installer = '3djuump-infinite-server'
				item.proxy_createdb_command = ['-creategenerationdb']
			elif item.proxy_role == 'static':
				lDefaultProxyBinaryFolder = '3DJuumpInfiniteStaticProxyx64'
				item.apache_conf_file_name = 'proxy'
				item.proxy_service_name = '3djuump-infinite-proxy'
				if sys.platform == 'win32' :
					item.proxy_exe_name = '3dJuumpInfiniteProxyService.exe'
					item.proxy_binary_installer = '3DJuump Infinite Static Proxy x64-setup-'
				else:
					item.proxy_exe_name = '3dJuumpInfiniteProxyService'
					item.proxy_binary_installer = '3djuump-infinite-proxy'
				item.proxy_createdb_command = ['-createproxydb','static']
			else:
				item.proxy_service_name = '3djuump-infinite-proxy'
				lDefaultProxyBinaryFolder = '3DJuumpInfiniteProxyx64'
				item.apache_conf_file_name = 'proxy'
				if sys.platform == 'win32' :
					item.proxy_exe_name = '3dJuumpInfiniteProxyService.exe'
					item.proxy_binary_installer = '3DJuump Infinite Proxy x64-setup-'
				else:
					item.proxy_exe_name = '3dJuumpInfiniteProxyService'
					item.proxy_binary_installer = '3djuump-infinite-proxy'
				item.proxy_createdb_command = ['-createproxydb','dynamic']
		if lIsInstallDirectory:
			if item.apache_conf_file_name != '':
				item.apache_conf_file_name = 'and' + item.apache_conf_file_name
			item.apache_conf_file_name = 'directory' + item.apache_conf_file_name
		item.apache_conf_file_name = '3djuump' + item.apache_conf_file_name
		
		# build auth variable for apache configuration
		if item.openidconnect['use_oidc_access_token']:
			# call oidc server to retrieve the configuration
			lHttpPool = requests.Session()
			if not item.openidconnect['verify_ssl_peer']:
				lHttpPool.verify = False
				requests.packages.urllib3.disable_warnings(requests.packages.urllib3.exceptions.InsecureRequestWarning)
			lProxy = {}
			if 'http_proxy' in item.openidconnect and item.openidconnect['http_proxy'] != None and item.openidconnect['http_proxy'] != '':
				lProxy['https'] = item.openidconnect['http_proxy']
			
			lJwksUri = None
			
			try:
				lGetRes = lHttpPool.get(item.openidconnect['configuration_endpoint'],proxies=lProxy)
				if lGetRes.status_code != 200:
					raise Exception('Fail to retrieve openid connect configuration')
				lOidcConf = json.loads(lGetRes.text);
				lJwksUri = lOidcConf['jwks_uri']
			except Exception as e:
				if not 'jwks_uri' in item.openidconnect or item.openidconnect['jwks_uri'] == '':
					raise Exception('/!\ fail to contact oidc server, to make an "offline install" set openidconnect.jwks_uri attribute')
				lJwksUri = item.openidconnect['jwks_uri']
			
			setattr(item,'auth_jwt_aud', item.openidconnect['oidc_access_token_aud']  if 'oidc_access_token_aud' in item.openidconnect else '')
			setattr(item,'auth_jwks_url',lJwksUri)
			setattr(item,'auth_jwks_ignore_invalid_cert', 'Off' if item.openidconnect['verify_ssl_peer'] is True else 'On' )
			setattr(item,'auth_jwks_proxy',item.openidconnect['http_proxy'] if 'http_proxy' in item.openidconnect and not item.openidconnect['http_proxy'] is None else '')
			
			# mod_infinite does not support HS* so remove them
			lAlgsForApache = []
			for a in item.openidconnect['allowed_jwt_alg']:
				if a in ['HS256','HS384','HS512']:
					continue
				lAlgsForApache.append(a)
			setattr(item,'auth_jwt_algs',' '.join(lAlgsForApache))
		else:
			setattr(item,'auth_jwt_aud',item.directory_public_url)
			setattr(item,'auth_jwks_url',item.directory_public_url + '/api/.well-known/jwks.json')
			setattr(item,'auth_jwks_ignore_invalid_cert', 'Off' if item.verify_ssl_peer else 'On' )
			setattr(item,'auth_jwks_proxy','')
			setattr(item,'auth_jwt_algs','RS256')
		
		# build inferred path if they were not specified
		if not 'inferred' in lConf:
			item._from_basepath = True
			
			lInferredPath = [
				('apache_log_folder', 'install_basepath' ,'current/www/logs'),
				('apache_folder','binary_basepath','Apache'),
				('auth_pwd_file', 'install_basepath' ,'current/ssl/apache.pwd'),
				('ssl_folder', 'install_basepath' ,'current/ssl'),
				('postgres_data_folder', 'install_basepath' ,'current/postgresql'),
				('postgres_folder','binary_basepath','PostgreSQL'),
				('apache_empty_root_folder', 'install_basepath' ,'current/www/empty/public'),
				]
			if lIsInstallDirectory:
				lInferredPath = lInferredPath + [
					('directory_log_folder', 'install_basepath' ,'current/service/logs'),
					('directory_binary_folder','binary_basepath','3DJuumpInfiniteDirectoryx64'),
				]
			if lIsInstallProxy:
				lInferredPath = lInferredPath + [
					('elastic_data_folder', 'install_basepath' ,'current/elasticsearch/data'),
					('elastic_binary_folder','binary_basepath','ElasticSearch'),
					('elastic_log_folder', 'install_basepath' ,'current/elasticsearch/logs'),
					('proxy_data_folder', 'install_basepath' ,'current/service/3d'),
					('proxy_binary_folder','binary_basepath',lDefaultProxyBinaryFolder),
					('proxy_log_folder', 'install_basepath' ,'current/service/logs'),
					('proxy_eseditor_folder', 'install_basepath' ,'current/www/eseditor'),
					('infinite_postgres_plugins_folder','binary_basepath','3DJuumpInfinitePostgresPluginsx64'),
					('java_binary_folder','binary_basepath','Java'),
				]
			for (attrName,base,suffix) in lInferredPath :
				attrValue = getCurrentAttribute(item,attrName)
				if attrValue == '' or attrValue is None:
					setattr(item,attrName,os.path.join(getattr(item,base),suffix).replace('\\','/'))
				else :
					item._from_basepath = False
			
		else:
			item._from_basepath = False
		item.certificate_file = os.path.join(item.ssl_folder,'server.crt')
		item.privatekey_file = os.path.join(item.ssl_folder,'private.key')
		
